-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local ViewPropertiesRegistry = lazyRequire("core.definitions.ViewPropertiesRegistry")

local AbstractListView = ViewClass('AbstractListView')
local IteratingView = class(AbstractListView)

IteratingView.typeName = 'IteratingView'

-- The className of the inlined view fragment which will be iterated.
IteratingView.iterationFragment = nil

-- The name of the ops key containing the data which will be iterated.
IteratingView.dataOpsKeyword = 'data'
-- TODO (pcowburn) Provided for backwards compatibility until all apps have
-- switched over to use the non-underscore prefixed version.
IteratingView.legacyDataOpsKeyword = '__data'

-- IteratingViews have to have a container node as their root element so that
-- the iteratee views have something to be attached to.
IteratingView._mustHaveRootElement = true


function IteratingView:createResources()
	self._iterationViewDefinition = self:_getIterateeDefinition()
	self._iterationViews = {}

	return AbstractListView.createResources(self)
end

function IteratingView:_getIterateeDefinition()
	local className = self.iterationFragment
	assert(className ~= nil, 'Please ensure iterationFragment is set on your IteratingView')

	local definition = self:getFragmentDefinition(className)
	assert(definition ~= nil, 'No fragment is registered with the className ' .. className)

	return definition
end

function IteratingView:preflight(ops)
	AbstractListView.preflight(self, ops)

	local iterationOps = self:_getIterationOps(ops)
	if iterationOps ~= nil then
		self:_preflightIterationOpsData(ops, iterationOps)
	end
end

function IteratingView:_getIterationOps(ops)
	local iterationOpsContainer = ops:getChild(self.dataOpsKeyword)
							   or ops:getChild(self.legacyDataOpsKeyword)

	if iterationOpsContainer ~= nil then
		-- The actual objects of data we're going to be creating iterated views
		-- for are the ones inside the __data array, so that means its children.
		return iterationOpsContainer:getChildren()
	else
		return nil
	end
end

function IteratingView:_preflightIterationOpsData(ops, iterationOps)

	for index, datumOp in pairs(iterationOps) do
		local type = datumOp:getType()

		if type == OpsContainer.OPERATION_CREATE then
			self:_preflightCreateOperation(index, datumOp)

		elseif type == OpsContainer.OPERATION_UPDATE then
			self:_preflightUpdateOperation(index, datumOp)

		-- Note that there's no OpsContainer.OPERATION_DELETE branch for this
		-- if statement - the child element that the op relates to is just going
		-- to be removed when update() is called, so there's nothing to preflight.
		end

		-- Copy the op into an aggregate that can be applied to the child when
		-- update() is finally called. This then means that the one of the 'apply'
		-- methods, i.e. _applyCreateOperation(), _applyDeleteOperation() or
		-- _applyUpdateOperation() will be invoked when update() is called later.
		ops:cloneToAggregate(index, datumOp)
	end

end

function IteratingView:_preflightCreateOperation(index, datumOp)
	-- Create a new instance of the iteration view and instruct it to create
	-- resources and forward the child op that relates to it.
	local instance = context:getViewFactory():createViewWithDefinition(
		self._iterationViewDefinition,
		self:getMediationGroup())

	-- Node index + 1 adjustment because the first child is the ContainerNode
	self:setChildElement(index+1, instance)

	instance:preflight(datumOp)

	context:getNodesRegistry():add(
		instance:getRootNode():getSceneNode():getId(),
		instance)

	-- Add it to the iterationViews list at the correct index.
	if #self._iterationViews > index and self._iterationViews[index] == nil then
		-- If we have: {a, b, c, nil, d} and try to insert a value in the 'nil'
		-- position, table.insert would end up shifting nil to 'd' and overwriting
		-- its value, so this "if" branch avoids this behaviour.
		self._iterationViews[index] = instance
	else
		table.insert(self._iterationViews, index, instance)
	end
end

function IteratingView:_preflightUpdateOperation(index, datumOp)
	local instance = self._iterationViews[index]

	if instance ~= nil then
		instance:preflight(datumOp)
	else
		self:_error('Could not retrieve child view to update')
	end
end

function IteratingView:_postApplyAggregates()
--Do nothing
end

function IteratingView:update(ops)
	AbstractListView.update(self, ops)

	local addedOrRemovedChildren = self:_applyAggregates(ops)
	self:_postApplyAggregates()

	if addedOrRemovedChildren then
		self:_setElementsPosition()
	end
end

function IteratingView:_setElementsPosition(options)
	if #self._iterationViews > 0 then
		self:getProperty("iterator"):iterate(self._iterationViews)
	end
end

function IteratingView:_applyAggregates(ops)
	local addedOrRemovedChildren = false
	local aggregateIndices = ops:getAggregateIndices()

	-- Elements are created/deleted later, as it would change
	-- self._iterationViews's size and break the iteration
	local createOps = {}
	local deleteOps = {}
	for i = 1, #aggregateIndices do
		local index = aggregateIndices[i]
		local aggregate = ops:getAggregate(index)
		local type = aggregate:getType()
		local iterationView = self._iterationViews[index]

		if iterationView then
			if type == OpsContainer.OPERATION_CREATE then
				table.insert(createOps,
					{
						index = index,
						iterationView = iterationView,
						aggregate = aggregate
					})
			elseif type == OpsContainer.OPERATION_UPDATE then
				self:_applyUpdateOperation(index, iterationView, aggregate)
			elseif type == OpsContainer.OPERATION_DELETE then
				table.insert(deleteOps,
						{
							index = index,
							iterationView = iterationView,
							aggregate = aggregate
						})
			end
		else
			log.error('No iteration view exists at index ' .. index)
		end
	end

	for i, e in ipairs(createOps) do
		self:_applyCreateOperation(e.index, e.iterationView, e.aggregate)
		addedOrRemovedChildren = true
	end

	-- Apply delete ops in reverse order, to avoid shifting the elements
	-- and breaking the iteration
	table.sort(deleteOps, function(a, b) return a.index > b.index end)
	for i, e in ipairs(deleteOps) do
		self:_applyDeleteOperation(e.index, e.iterationView, e.aggregate)
		addedOrRemovedChildren = true
	end

	return addedOrRemovedChildren
end

function IteratingView:_applyCreateOperation(index, iterationView, aggregateOp)
	local selfNode = self:getRootNode()
	local childNode = iterationView:getRootNode()

	assert(selfNode ~= nil, '_applyCreateOperation: selfNode must not be nil')
	assert(childNode ~= nil, '_applyCreateOperation: childNode must not be nil')

	context:getNodesRegistry():add(
		iterationView:getRootNode():getSceneNode():getId(),
		iterationView)

	self:_setupWatchers(iterationView)

	-- Node index + 1 adjustment because the first child is the ContainerNode
	self:setChildElement(index+1, iterationView)
	-- Note index - 1 adjustment because SceneNode child vectors are zero indexed.
	context:getSceneNodeOpsQueue():queueOp(
		selfNode:getSceneNode(),
		"addChildAt",
		childNode:getSceneNode(),
		index - 1)

	self:_applyUpdateOperation(index, iterationView, aggregateOp)
end

function IteratingView:_setupWatchers(iterationView)
	iterationView:addWatcher("width", self)
	iterationView:appendWatchers("width", self:getWatchers("width"))
	iterationView:addWatcher("height", self)
	iterationView:appendWatchers("height", self:getWatchers("height"))
end

function IteratingView:onNodePropertyChange(propertiesChanged)
	AbstractListView.onNodePropertyChange(self, propertiesChanged)
	if self:getProperty("iterator"):doPropertyChangesAffectIteration(propertiesChanged) then
		self:_iterateElementsOnNodePropertyChange(propertiesChanged)
	end
end

function IteratingView:_iterateElementsOnNodePropertyChange()
	self:_setElementsPosition()
end

function IteratingView:_applyDeleteOperation(index, iterationView, aggregateOp)
	local selfNode = self:getRootNode()
	local childNode = iterationView:getRootNode()

	assert(selfNode ~= nil, '_applyDeleteOperation: selfNode must not be nil')
	assert(childNode ~= nil, '_applyDeleteOperation: childNode must not be nil')

	-- Note: index+1 ajustment because the first child is the ContainerNode
	self:setChildElement(index+1, nil)

	context:getSceneNodeOpsQueue():queueOp(
		selfNode:getSceneNode(),
		"removeChild",
		childNode:getSceneNode())

	iterationView:dispose()

	table.remove(self._iterationViews, index)
end

function IteratingView:_applyUpdateOperation(index, iterationView, aggregateOp)
	iterationView:update(aggregateOp)
end

function IteratingView:getIterationView(index)
	return self._iterationViews[index]
end

function IteratingView:dispose()
	for index = 1, #self._iterationViews do
		self._iterationViews[index]:dispose()
	end
	self._iterationViews = {}

	return AbstractListView.dispose(self)
end

return IteratingView
