-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local ViewPropertiesRegistry = plugins.views:lazyRequire("core.definitions.ViewPropertiesRegistry")
local IteratorOptions = plugins.views:lazyRequire("properties.iterator.IteratorOptions")

local Viewport = plugins.views:lazyRequire('views.infiniteList.view.Viewport')
local ViewPool = plugins.views:lazyRequire('views.infiniteList.view.ViewPool')
local ViewportViewsManager = plugins.views:lazyRequire('views.infiniteList.view.ViewportViewsManager')
local FixedHighlightViewportPositioner = plugins.views:lazyRequire('views.infiniteList.view.positioning.FixedHighlightViewportPositioner')

local DataChunk = plugins.views:lazyRequire('views.infiniteList.data.DataChunk')
local DataBuffer = plugins.views:lazyRequire('views.infiniteList.data.DataBuffer')
local LookaheadDataBufferPositioner = plugins.views:lazyRequire('views.infiniteList.data.positioning.LookaheadDataBufferPositioner')

local RetainedObjectUtils = plugins.views:lazyRequire('utils.RetainedObjectUtils')
local KeyRepeatInterceptor = plugins.views:lazyRequire('utils.KeyRepeatInterceptor')
local ScrollingTimer = plugins.views:lazyRequire('utils.ScrollingTimer')
local ListScroller = plugins.views:lazyRequire('utils.ListScroller')
local IndexTranslator = plugins.views:lazyRequire('views.IndexTranslator')

local AbstractListView = ViewClass('AbstractListView')
local InfiniteListView = class(AbstractListView, function(self, handle)
	AbstractListView._ctor(self, handle)

	self._chunksToRequest = {}
	self._isMediatorReady = false
	self._isInitialized = false
	self._hasInitializedPosition = false
	context:getTickableRegistry():addTickable(self)
end)

InfiniteListView.fastScrollTickSignalName = 'fastScrollTick'
InfiniteListView.fastScrollStopSignalName = 'fastScrollStopped'
InfiniteListView.fastScrollStartSignalName = 'fastScrollStarted'

InfiniteListView.typeName = 'InfiniteListView'
InfiniteListView.highlightedTag = 'highlighted'

-- The className of the inlined view fragment which will be created for each
-- item in the list.
InfiniteListView.listItemFragment = nil

-- InfiniteListViews have to have a container node as their root element so that
-- the list item views have something to be attached to.
InfiniteListView._mustHaveRootElement = true

InfiniteListView._requiredProperties =
{
	"viewPoolSize",
	"viewportSize",
	"chunkSize",
	"lookAhead"
}

InfiniteListView.propertyDefinitions = ViewPropertiesRegistry.mergePropertyDefinitions(
	AbstractListView.propertyDefinitions,
	{
		transitionDuration =
		{
			default = 100
		},
		easingFunction =
		{
			default = "outQuad",
			onChange = function(carouselView, oldValue, newValue)
				if newValue then
					animation.loadFunction(newValue)
				end
			end
		},
		selectedPos =
		{
			default = 1,
			onChange = function(infiniteListView)
				infiniteListView:_initialiseComponentsIfReadyOrCallCallbackIfInitialised(function()
					infiniteListView:_createViewportPositioner()
					infiniteListView:_setElementsPosition()
				end)
			end
		},
		initialPos =
		{
			default = 1,
			onChange = function(infiniteListView)
				infiniteListView:_initialiseComponentsIfReadyOrCallCallbackIfInitialised(function()
					infiniteListView:_resetViewport()
				end)
			end
		},
		defaultModel =
		{
			default = {}
		},
		viewportSize =
		{
			default = nil,
			onChange = function(infiniteListView)
				infiniteListView:_initialiseComponentsIfReadyOrCallCallbackIfInitialised(function()
					infiniteListView:_resetViewport()
				end)
			end
		},
		viewPoolSize =
		{
			default = nil,
			onChange = function(infiniteListView)
				infiniteListView:_initialiseComponentsIfReadyOrCallCallbackIfInitialised(function()
					infiniteListView:_resetViewport()
				end)
			end
		},
		chunkSize =
		{
			default = nil,
			onChange = function(infiniteListView)
				infiniteListView:_initialiseComponentsIfReadyOrCallCallbackIfInitialised(function()
					infiniteListView:_createDataBuffer()
					infiniteListView:_resetViewport()
				end)
			end
		},
		lookAhead =
		{
			default = nil,
			onChange = function(infiniteListView)
				infiniteListView:_initialiseComponentsIfReadyOrCallCallbackIfInitialised(function()
					infiniteListView:_createDataBufferPositioner()
				end)
			end
		},
		viewportBackPadding =
		{
			default = 0,
			onChange = function(infiniteListView)
				infiniteListView:_initialiseComponentsIfReadyOrCallCallbackIfInitialised(function()
					infiniteListView:_resetViewport()
				end)
			end
		},
		scrollingAnimation =
		{
			default =
			{
				easingFunctionName = "linear",
				duration = 300
			},
			onChange = function(infiniteListView, oldValue, newValue)
				infiniteListView._listScroller:setAnimation(newValue)
			end
		},
		scrollingDuration =
		{
			onChange = function(infiniteListView, oldValue, newValue)
				log.warn("InfiniteListView: the 'scrollingDuration' view property " ..
						"is now DEPRECATED; please define your scrolling animation in " ..
						"full using the 'scrollingAnimation' view property.")
				local scrollingAnimation = infiniteListView:getProperty("scrollingAnimation")
				infiniteListView._listScroller:setAnimation(
					{
						easingFunctionName = scrollingAnimation.easingFunctionName,
						duration = newValue
					})
			end
		},
		interceptInput =
		{
			onChange = function()
				log.warn("InfiniteListView: the 'interceptInput' view property " ..
						"is now DEPRECATED; setting it will have no effect.")
			end
		}
	}
)

function InfiniteListView:tick()
	self:_requestQueuedChunks()
end

function InfiniteListView:_requestQueuedChunks()
	local signal = self:getSignal('chunkDataRequired')
	if signal then
		for i, chunk in ipairs(self._chunksToRequest) do
			signal:dispatch(self:_buildSignalParamsFromChunk(chunk))
		end
		self._chunksToRequest = {}
	end
end

function InfiniteListView:_setupWatchers(childView)
	childView:addWatcher("width", self)
	childView:appendWatchers("width", self:getWatchers("width"))
	childView:addWatcher("height", self)
	childView:appendWatchers("height", self:getWatchers("height"))
end

function InfiniteListView:createResources()
	AbstractListView.createResources(self)

	self._listItemViewDefinition = self:_getListItemViewDefinition()
	self._listItemViews = {}

	local signal = self:addBubblingSignal("internalFocusGained")
	signal:add(self._onFocusGained, self)

	local signal = self:addBubblingSignal("internalFocusLost")
	signal:add(self._onFocusLost, self)

	self:_initListScroller()

	self:addSignal(InfiniteListView.fastScrollStartSignalName)
	self:addSignal(InfiniteListView.fastScrollStopSignalName)
	self:addSignal(InfiniteListView.fastScrollTickSignalName)

	return true
end

function InfiniteListView:_initialiseComponentsIfReadyOrCallCallbackIfInitialised(updateCallback)
	if self._isInitialized then
		updateCallback()
	elseif self:_isReadyToInitialize() then
		self:_initialiseComponents()
	end
end

function InfiniteListView:_resetViewport()
	self:_createViewportViewsManager()
	self:_hookDataBufferToViewportViewsManager()
	self:_initialiseInfiniteList()
end 

function InfiniteListView:_initListScroller()
	local initialScrollingAnimation = self:getProperty("scrollingAnimation")
	self._scrollingIteratorOptions = IteratorOptions.new({})
	self._listScroller = self:_createListScroller(initialScrollingAnimation)
	if initialScrollingAnimation then
		self._listScroller:setAnimation(initialScrollingAnimation)
	end
end

function InfiniteListView:_createListScroller()
	return ListScroller.new(
	{
		keyRepeatInterceptor = KeyRepeatInterceptor.new(300),
		scrollingTimer = ScrollingTimer.new(),
		callbackContext = self,
		callbacks =
		{
			getDirectionForKey = self._getDirectionForKey,
			hasElementsInDirection = function() return true end,
			scroll = self._scroll,
			onStartScrolling = self._startScrolling,
			onStopScrolling = self._stopScrolling
		}
	})
end

function InfiniteListView:onMediatorReady()
	self._isMediatorReady = true

	if self:_isReadyToInitialize() then
		self:_initialiseComponents()
	end
end

function InfiniteListView:_isReadyToInitialize()
	if not self._isMediatorReady then
		return false
	end

	for _, propertyName in ipairs(self._requiredProperties) do
		if self:getProperty(propertyName) == nil then
			return false
		end
	end

	return true
end

function InfiniteListView:_initialiseComponents()
	self:_createViewportPositioner()
	self:_createDataBuffer()
	self:_createDataBufferPositioner()
	self:_createViewportViewsManager()
	self:_initialiseInfiniteList()
	self:_setElementsPosition()

	self._isInitialized = true
end

function InfiniteListView:_initialiseInfiniteList()
	self:_hookViewportViewsManagerToDataBuffer()
	self:_hookDataBufferToViewportViewsManager()

	local viewport = self._viewportViewsManager:getViewport()

	self._viewportPositioner:setHighlightedIndex(viewport, self:getProperty("initialPos"))
	self._viewportViewsManager:calculateTargetViewPositionsForNextUpdateCycle()

	self:_moveHighlightedTagToIndex(
			viewport:getPreviousHighlightedIndex(),
			viewport:getHighlightedIndex())

	for i, view in ipairs(self:getViewportViewsManager():getVisibleViews()) do
		self:_setupWatchers(view)
	end
end

function InfiniteListView:_getListItemViewDefinition()
	local className = self.listItemFragment
	assert(className ~= nil, 'Please ensure listItemFragment is set on your InfiniteListView')

	local definition = self:getFragmentDefinition(className)
	assert(definition ~= nil, 'No fragment is registered with the className ' .. className)

	return definition
end

function InfiniteListView:_createViewportViewsManager()
	if self._viewportViewsManager then
		local viewport = self._viewportViewsManager:getViewport()
		self:_moveHighlightedTagToIndex(viewport:getHighlightedIndex(), nil)

		self._viewportViewsManager:dispose()
		viewport:dispose()
	end

	self:_createOrResizeViewPool()

	local viewport = Viewport.new(
			self:getProperty("viewportSize"), self:getProperty("viewportBackPadding"))

	self._viewportViewsManager = ViewportViewsManager.new(
			self._viewPool,
			viewport,
			self)

	viewport.positionChanged:add(function()
		self:_moveHighlightedTagToIndex(
				viewport:getPreviousHighlightedIndex(),
				viewport:getHighlightedIndex())

		self._dataBufferPositioner:repositionDataBuffer(self._dataBuffer, viewport)
	end)
end

function InfiniteListView:_createOrResizeViewPool()
	local viewPoolSize = self:getProperty("viewPoolSize")

	if self._viewPool == nil then
		self._viewPool = ViewPool.new(
			viewPoolSize,
			self._listItemViewDefinition,
			self:getMediationGroup())
	else
		self._viewPool:resize(viewPoolSize)
	end
end

function InfiniteListView:_createViewportPositioner()
	self._viewportPositioner = FixedHighlightViewportPositioner.new(
			self:getProperty("selectedPos"))
end

function InfiniteListView:_createDataBuffer()
	self._dataBuffer = DataBuffer.new(self:getProperty("chunkSize"))

	self:_addForwardingSignalForChunkDataRequired()
end

function InfiniteListView:_createDataBufferPositioner()
	self._dataBufferPositioner =
			LookaheadDataBufferPositioner.new(self:getProperty("lookAhead"))
end

function InfiniteListView:_setElementsPosition()
	if self:getViewportViewsManager()
			and (self:getViewportViewsManager():getNumVisibleViews() > 0) then
		self:_repositionViewport(
			self:getProperty("iterator"),
			IteratorOptions.new({})
		)
	end
end

function InfiniteListView:_hookViewportViewsManagerToDataBuffer()
	if not self._dataBuffer.chunkDataReceived:has(self._onChunkDataReceived, self) then
		self._dataBuffer.chunkDataReceived:add(self._onChunkDataReceived, self)
	end
end

function InfiniteListView:_onChunkDataReceived(chunk)
	self._viewportViewsManager:provideDataToViews(chunk, self._dataBuffer)
	self:_redraw()
end

function InfiniteListView:_hookDataBufferToViewportViewsManager()
	local viewportViewsManager = self._viewportViewsManager

	if not viewportViewsManager.viewAddedToViewport:has(
			self._onViewAddedToViewport, self) then
		viewportViewsManager.viewAddedToViewport:add(
				self._onViewAddedToViewport, self)
	end

	if not viewportViewsManager.viewRemovedFromViewport:has(
			self._onViewRemovedFromViewport, self) then
		viewportViewsManager.viewRemovedFromViewport:add(
				self._onViewRemovedFromViewport, self)
	end
end

function InfiniteListView:_onViewAddedToViewport(view)
	view.relativeIndexWithinDataSet = view.indexWithinDataSet

	local data = self._dataBuffer:getItemByIndex(view.relativeIndexWithinDataSet)
	if data ~= nil then
		self._viewportViewsManager:removeViewWaitingForData(view)
		view:updateModel(data)
	else
		self:_provideDefaultModel(view)
		self._viewportViewsManager:addViewWaitingForData(view)
	end
end

function InfiniteListView:_onViewRemovedFromViewport(view)
	self:_provideDefaultModel(view)
end

function InfiniteListView:_provideDefaultModel(view)
	local defaultModel = self:getProperty("defaultModel")
	if defaultModel ~= nil then
		view:updateModel(defaultModel)
	end
end

function InfiniteListView:_addForwardingSignalForChunkDataRequired()
	if not self:getSignal('chunkDataRequired') then
		self:addSignal('chunkDataRequired')
	end

	self._dataBuffer.chunkDataRequired:add(function(chunk)
		table.insert(self._chunksToRequest, chunk)
	end)
end


function InfiniteListView:_buildSignalParamsFromChunk(chunk)
	-- TODO (pcowburn) it'd be nicer if something built this vector up for
	-- us (perhaps based on a plain Lua table) rather than doing this manually
	-- whenever we want to dispatch a Signal to C++.
	local signalParams = ReflectableValueVectorUserData.new()

	local viewId = ReflectableValueUserData.new()
	viewId:setInt(self:getHandle():getUuid())

	local viewName = ReflectableValueUserData.new()
	if self:getName() then
		viewName:setString(self:getName())
	else
		viewName:setString("")
	end

	local startIndex = ReflectableValueUserData.new()
	startIndex:setInt(chunk:getStartRequestIndex() - 1)

	local endIndex = ReflectableValueUserData.new()
	endIndex:setInt(chunk:getEndRequestIndex())

	signalParams:push(viewId)
	signalParams:push(viewName)
	signalParams:push(startIndex)
	signalParams:push(endIndex)

	return signalParams
end

function InfiniteListView:getDataBuffer()
	return self._dataBuffer
end

function InfiniteListView:getViewportViewsManager()
	return self._viewportViewsManager
end

function InfiniteListView:updateModel(model)
	-- TODO(Bruno) This '.restartList' is actually a "hack". We have to provide
	--             a way for views to explicitly reset their view model, e.g.
	--             viewHandle.resetModel() or viewHandle.resetList(). This
	--             should be dealt with in Jira IG-263.
	if model.restartList and self:_isReadyToInitialize() then
		self:_initialiseComponents()
	elseif self:_isChunkPackage(model) then
		self:_buildChunkAndAddToBuffer(model)
	end
end

function InfiniteListView:onNodePropertyChange(propertiesChanged)
	AbstractListView.onNodePropertyChange(self, propertiesChanged)

	-- A 'reposition' may have been queued for a scene.afterNextFlush while
	-- fast scrolling, so we must make sure we won't "jump in front" of it,
	-- otherwise weird animation artifacts may occur.
	if not self._queuedRepositioning and
			self:getProperty("iterator"):doPropertyChangesAffectIteration(
				propertiesChanged) then
		self:_iterateElementsOnNodePropertyChange()
	end
end

function InfiniteListView:_iterateElementsOnNodePropertyChange()
	-- We assume here that the first node property change signal
	-- is a result of the dimensions of the list item nodes being set,
	-- so we don't animate to the new positions; any subsequent adjustments
	-- to list item dimensions result in animated iterations.
	if not self._hasInitializedPosition then
		self:_setElementsPosition()
		self._hasInitializedPosition = true
	else
		self:_repositionViewport(
			self:getProperty("iterator"),
			IteratorOptions.new({
				easingFn = self:getProperty("easingFunction"),
				duration = self:getProperty("transitionDuration")
			})
		)
	end
end

function InfiniteListView:_isChunkPackage(model)
	return model.items ~= nil
	   and model.startIndex ~= nil
	   and model.endIndex ~= nil
end

function InfiniteListView:_buildChunkAndAddToBuffer(model)
	if model.totalDataSetSize ~= nil then
		self._dataBuffer:setTotalDataSetSize(model.totalDataSetSize)
	end

	-- Note + 1 as the model is coming from JS and so will be 0 indexed.
	local chunk = DataChunk.new(
			model.startIndex + 1,
			model.endIndex,
			model.items,
			RetainedObjectUtils.getObjectId(model))

	self._dataBuffer:addChunkData(chunk)
end

function InfiniteListView:preflight(ops)
	-- This is a no-op for InfiniteListView, as it takes its data directly from
	-- the updateModel() method.
end

function InfiniteListView:getFocusCandidate()
	local candidateView = self:getViewportViewsManager():getHighlightedView()
	if candidateView then
		return candidateView:getRootNode():getSceneNode()
	end
end

function InfiniteListView:interceptFocusGain(oldNode, newNode, direction, callingView)
	if self._listScroller and self._listScroller:isScrolling() then
		return nil
	end

	local selectedNode = newNode
	local nextInterceptor = self:getParent()
	local interceptCaller = self

	-- This prevents the user from being able to scroll through the
	-- infiniteList if there's no data to be seen (i.e. the view is still
	-- waiting for chunks to come back from the App)
	if self:_isNodeWaitingForChunkData(newNode) then
		local oldDirective =
				context:getNodesRegistry():getViewElement(oldNode:getId())

		selectedNode = oldNode
		nextInterceptor = oldDirective:getParent()
		interceptCaller = nil
	end

	if nextInterceptor then
		selectedNode = nextInterceptor:interceptFocusGain(
				oldNode, selectedNode, direction, interceptCaller)
	end

	if selectedNode == oldNode then
		return nil
	end

	return selectedNode
end

function InfiniteListView:_onFocusGained(focusedNode)
	local child = self:_iterativelyGetChildNodeWhichContainsNode(focusedNode)

	self:_updateHighlightedIndexBasedOnNewlyFocusedNode(child)

	if self._listScroller then
		self._listScroller:activate()
	end
end

function InfiniteListView:_onFocusLost(nodeWhichLostFocus)
        -- We need to deactivate the input interceptor when the list loses focus,
        -- (ie. a descendant loses focus) otherwise when the user press
        -- "left/right/whatever" in another list this one (which lost focus)
        -- will continue to move (as it'll still be intercepting the input)

        -- We do not have access to the newly focused node here, so we guard
        -- against disabling the list scroller when focus is removed during
        -- scrolling with an explicit check
	if self._listScroller and not self._listScroller:isScrolling() then
		self._listScroller:deactivate()
	end
end

function InfiniteListView:_isNodeWaitingForChunkData(node)
	local directive = context:getNodesRegistry():getViewElement(node:getId())
	return self._viewportViewsManager:isViewWaitingForData(directive)
end

function InfiniteListView:_updateHighlightedIndexBasedOnNewlyFocusedNode(newNode)
	local selfNode = self:getRootNode():getSceneNode()

	if newNode:hasParent() and newNode:getParent():getId() == selfNode:getId() then

		local nodeDirective =
				context:getNodesRegistry():getViewElement(newNode:getId())

		local highlightedIndex = nodeDirective.indexWithinDataSet

		self:_setHighlightedIndex(highlightedIndex)
	end
end

function InfiniteListView:_setHighlightedIndex(highlightedIndex, iteratorOptionsOverride)
	local previousIndex = self._viewportViewsManager:getViewport():getHighlightedIndex()

	if highlightedIndex ~= previousIndex then
		local viewport = self._viewportViewsManager:getViewport()

		self._viewportPositioner:setHighlightedIndex(viewport, highlightedIndex)
		self:_moveHighlightedTagToIndex(previousIndex, viewport:getHighlightedIndex())
		self:_redraw()

		if self._viewportPositioner:shouldReposition(viewport) then
			-- We need to give time for new nodes (added in "_redraw()"") to be
			-- styled, so the animation is queued only after the next scene tick
			self:_queueRepositionViewport(
					self:getProperty("iterator"),
					iteratorOptionsOverride or IteratorOptions.new({
						easingFn = self:getProperty("easingFunction"),
						duration = self:getProperty("transitionDuration")
					}))
		end
	end
end

function InfiniteListView:_queueRepositionViewport(iterator, iteratorOptions)
	self._queuedRepositioning = true

	scene.afterNextFlush(function()
		self:_repositionViewport(iterator, iteratorOptions)
		self._queuedRepositioning = false
	end)
end

function InfiniteListView:_repositionViewport(iterator, iteratorOptions)
	self._viewportPositioner:reposition(
		self._viewportViewsManager:getViewport(),
		self:getViewportViewsManager():getVisibleViews(),
		iterator,
		iteratorOptions)
end

function InfiniteListView:_moveHighlightedTagToIndex(fromIndex, toIndex)
	local fromView = self._viewportViewsManager:getView(fromIndex)
	if fromView then
		fromView:getRootNode():getSceneNode():removeTag(InfiniteListView.highlightedTag)
	end

	local toView = self._viewportViewsManager:getView(toIndex)
	if toView then
		toView:getRootNode():getSceneNode():addTag(InfiniteListView.highlightedTag)
	end
end

function InfiniteListView:_redraw()
	self._viewportViewsManager:calculateTargetViewPositionsForNextUpdateCycle()
	self._viewportViewsManager:updateViewPositionsWithinParentContainer()
end

function InfiniteListView:getListItemView(index)
	return self._viewportViewsManager:getViewAtIndex(index)
end

function InfiniteListView:dispose()
	context:getTickableRegistry():removeTickable(self)
	self._viewportViewsManager:getViewport():dispose()
	self._viewportViewsManager:dispose()
	self._dataBuffer:dispose()

	if self._listScroller then
		self._listScroller:dispose()
		self._listScroller = nil
	end

	if self._viewPool ~= nil then
		self._viewPool:dispose()
	end

	return AbstractListView.dispose(self)
end

function InfiniteListView:_startScrolling(direction)
	signalBus:getSignal("requestFocus"):dispatch(
			nil, { updateAncestorTags = false })
	self:getSignal(InfiniteListView.fastScrollStartSignalName):dispatch()
end

function InfiniteListView:_scroll(direction, scrollingIteratorOptions)
	local viewToHighlight =
			self:_getViewInDirection(direction)

	if self._viewportViewsManager:isViewAvailable(viewToHighlight)
			and self:_viewIsFocusable(viewToHighlight) then

		self:_setHighlightedIndex(viewToHighlight, scrollingIteratorOptions)

		self:_dispatchTickSignal(viewToHighlight)
	end
end

function InfiniteListView:_dispatchTickSignal(viewIndex)
	local tickSignal =
			self:getSignal(InfiniteListView.fastScrollTickSignalName)
	local signalParams = ReflectableValueVectorUserData.new()
	local view = self._viewportViewsManager:getView(viewIndex)

	if view and view.getRetainedObjectId then
		local viewModelParam = ReflectableValueUserData.new()
		viewModelParam:setId(view:getRetainedObjectId())
		signalParams:push(viewModelParam)
	end

	tickSignal:dispatch(signalParams)
end

function InfiniteListView:_stopScrolling(scrollingDirection)
	self:getSignal(InfiniteListView.fastScrollStopSignalName):dispatch()

	local highlightedIndex =
			self._viewportViewsManager:getViewport():getHighlightedIndex()

	local nextViewIdx =
			self:_getViewInDirection(scrollingDirection)
	local nextView = self._viewportViewsManager:getView(nextViewIdx)

	local viewToFocus

	if (nextViewIdx ~= highlightedIndex)
			and not self._viewportViewsManager:isViewWaitingForData(nextView)
			and self:_viewIsFocusable(nextViewIdx) then
		self:_setHighlightedIndex(nextViewIdx)
		viewToFocus = nextView
	else
		viewToFocus = self._viewportViewsManager:getView(highlightedIndex)
	end

	if viewToFocus then
		signalBus:getSignal("requestFocus"):dispatch(
				viewToFocus:getRootNode():getSceneNode(),
				{ updateAncestorTags = false })
	end
end

function InfiniteListView:_viewIsFocusable(viewIndex)
	local view = self._viewportViewsManager:getView(viewIndex)
	if not view then
		return false
	end
	return view:getRootNode():getSceneNode():getFocusable()
end

function InfiniteListView:_getViewInDirection(direction)
	local highlightedIndex =
			self._viewportViewsManager:getViewport():getHighlightedIndex()
	local idxOfFirstView = self._viewportViewsManager:getIndexOfFirstView()
	local idxOfLastView = self._viewportViewsManager:getIndexOfLastView()

	local viewToHighlight
	if self:getProperty("iterator"):isForward(direction) then
		viewToHighlight = math.min(math.floor(highlightedIndex+1), idxOfLastView)
	else
		viewToHighlight = math.max(math.floor(highlightedIndex-1), idxOfFirstView)
	end

	return viewToHighlight
end

return InfiniteListView
