-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ElementView = ViewClass("ElementView")
local DetachingView = class(ElementView)

DetachingView.typeName = 'DetachingView'

-- The name of the ops keyword which dictates if the view is attached or not.
DetachingView.attachedStateOpsKeyword = 'attached'
-- TODO (pcowburn) Provided for backwards compatibility until all apps have
-- switched over to use the non-underscore prefixed version.
DetachingView.legacyAttachedStateOpsKeyword = '__attached'

-- Name of the view that is to be attached/detached.
DetachingView.detachingFragment = nil

-- Index that the detachable child view will be placed at in the _childElements
-- array.
DetachingView.detachableChildElementIndex = 2

-- DetachingViews have to have a container node as their root element so that
-- the detachable child view has something to be attached to.
DetachingView._mustHaveRootElement = true

DetachingView.elements =
{
	{
		className = 'ContainerNodeDirective',
		type = ElementView.TYPE_DIRECTIVE,
		parent = nil
	}
}

function DetachingView:createResources()
	ElementView.createResources(self)

	self:_createDetachableChildElement()

	return true
end

function DetachingView:_createDetachableChildElement()
	local viewFactory = context:getViewFactory()
	local definition = self:_getDetachableChildElementDefinition()
	local instance = viewFactory:createViewWithDefinition(
		definition,
		self:getMediationGroup())

	self._detachableChildElement = instance

	self:setChildElement(self.detachableChildElementIndex, instance)
end

function DetachingView:_getDetachableChildElementDefinition()
	local className = self.detachingFragment
	assert(className ~= nil, 'Please ensure detachingFragment is set on your DetachingView')

	local definition = self:getFragmentDefinition(className)
	assert(definition ~= nil, 'No fragment is registered with the className ' .. className)

	return definition
end

function DetachingView:preflight(ops)
	self._detachableChildElement:preflight(ops)

	return ElementView
end

function DetachingView:update(ops)
	-- Decide if we need to attach/detach.
	local attachState = self:_retrieveAttachStateFromOps(ops)

	self._detachableChildElement:update(ops)

	-- Toggle whether or not the child view is attached based on the attachState
	-- that was supplied. Note that the conditions check that the value is either
	-- true or false - i.e. if it's nil then nothing will change.
	if attachState == true then
		self:_attachChildElement()
	elseif attachState == false then
		self:_detachChildElement()
	end
end

function DetachingView:_retrieveAttachStateFromOps(ops)
	local op = ops:getLocal(self.attachedStateOpsKeyword)
			or ops:getLocal(self.legacyAttachedStateOpsKeyword)

	if op == nil then
		return nil
	else
		return op.value
	end
end

function DetachingView:_attachChildElement()
	local selfNode = self:getRootNode()
	local childNode = self._detachableChildElement:getRootNode()

	assert(selfNode ~= nil, '_attachChildElement: selfNode must not be nil')
	assert(childNode ~= nil, '_attachChildElement: childNode must not be nil')

	context:getSceneNodeOpsQueue():queueOp(
		selfNode:getSceneNode(),
		"addChild",
		childNode:getSceneNode())
end

function DetachingView:_detachChildElement()
	local selfNode = self:getRootNode()
	local childNode = self._detachableChildElement:getRootNode()

	assert(selfNode ~= nil, '_detachChildElement: selfNode must not be nil')
	assert(childNode ~= nil, '_detachChildElement: childNode must not be nil')

	context:getSceneNodeOpsQueue():queueOp(
		selfNode:getSceneNode(),
		"removeChild",
		childNode:getSceneNode())
end

function DetachingView:getDetachableChildElement()
	return self:getChildElement(self.detachableChildElementIndex)
end

return DetachingView
