-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local SignalInterceptor = plugins.lua:lazyRequire("signals.SignalInterceptor")
local ScrollingTimer = plugins.views:lazyRequire("utils.ScrollingTimer")
local KeyRepeatInterceptor = plugins.views:lazyRequire('utils.KeyRepeatInterceptor')
local ListScroller = plugins.views:lazyRequire('utils.ListScroller')
local NodeBumper = plugins.views:lazyRequire('utils.NodeBumper')
local ViewPropertiesRegistry = plugins.views:lazyRequire("core.definitions.ViewPropertiesRegistry")
local IteratorOptions = plugins.views:lazyRequire('properties.iterator.IteratorOptions')

local IteratingView = ViewClass('IteratingView')
local CarouselView = class(IteratingView, function(self)
	IteratingView._ctor(self)
	self._hasInitializedPosition = false
end)

local BehaviourFactory = plugins.views:lazyRequire('views.behaviours.BehaviourFactory')

CarouselView.typeName = 'CarouselView'

-- The property which describes the active index within the iterator.
CarouselView.indexOpsKeyword = 'index'
-- TODO (pcowburn) Provided for backwards compatibility until all apps have
-- switched over to use the non-underscore prefixed version.
CarouselView.legacyIndexOpsKeyword = '__index'
CarouselView.activeItemTag = 'carousel-view-active-item'

-- The op name passed to the active child view to let it know that it is active.
CarouselView.activeOpsName = "active"
CarouselView.defaultScrollingInterval = 300

CarouselView.boundsReachedBehaviourValues =
{
	LOCK_FOCUS = "LOCK_FOCUS",
	WRAP_FOCUS = "WRAP_FOCUS",
	BUMP = "BUMP"
}

CarouselView.propertyDefinitions = ViewPropertiesRegistry.mergePropertyDefinitions(
	IteratingView.propertyDefinitions,
	{
		transitionDuration =
		{
			default = 100
		},
		easingFunction =
		{
			default = "outQuad",
			onChange = function(carouselView, oldValue, newValue)
				if newValue then
					animation.loadFunction(newValue)
				end
			end
		},
		enableBumpOnBoundsReached =
		{
			default = false,
			onChange = function()
				log.warn("CarouselView: 'enableBumpOnBoundsReached' view property is DEPRECATED; "..
					"please consider using boundsReachedBehaviour = 'BUMP'")
			end
		},
		bumpDistance =
		{
			default = 30
		},
		bumpTransitionDuration =
		{
			default = 200
		},
		flexibleSelectedPos =
		{
			default = false,
			onChange = function(carouselView)
				carouselView:_setElementsPosition()
			end
		},
		selectedPos =
		{
			default = 1,
			onChange = function(carouselView)
				carouselView:_setElementsPosition()
			end
		},
		selectedPosMaxOffset =
		{
			default = nil
		},
		spacingAfterHighlightedItem =
		{
			default = 0,
			onChange = function(carouselView, oldValue, newValue)
				carouselView:_setElementsPosition()
			end
		},
		keyGoLeft =
		{
			default = nil,
			onChange = function(carousel, oldValue, newValue)
				carousel._keyCommandMap[newValue] = 'left'
				if oldValue then
					carousel._keyCommandMap[oldValue] = nil
				end
			end
		},
		keyGoRight =
		{
			default = nil,
			onChange = function(carousel, oldValue, newValue)
				carousel._keyCommandMap[newValue] = 'right'
				if oldValue then
					carousel._keyCommandMap[oldValue] = nil
				end
			end
		},
		behaviour =
		{
			default = "DefaultBehaviour",
			onChange = function(carousel, oldValue, newValue)
				carousel._behaviour = BehaviourFactory.createBehaviour(
					newValue)
			end
		},
		scrollingAnimation =
		{
			default = nil,
			onChange = function(carouselView, oldValue, newValue)
				carouselView:_updateScrollingAnimation(newValue)
			end
		},
		-- The 'boundsReachedBehaviour' property specifies the behaviour when the
		-- user reaches the bounds of the list view
		-- If set to LOCK_FOCUS, the focus has to stay in the view
		-- If set to WRAP_FOCUS, the focus wraps to the end/beginning of the list
		-- If set to BUMP, the view will have its focus locked and will have a bump animation, otherwise
		-- No special behaviour will be executed when the bounds are reached.
		boundsReachedBehaviour =
		{
			default = nil,
			onChange = function(carouselView, oldValue, newValue)
				carouselView:_validateBoundsReachedBehaviourUsage(newValue)
				if newValue == CarouselView.boundsReachedBehaviourValues.LOCK_FOCUS then
					-- should move the boundsReachedBehaviour property in AbstractListView
					-- sim: https://sim.amazon.com/issues/IGN-1113
					carouselView:setProperty("lockFocus", true)
				end
			end
		}
	}
)

function CarouselView:createResources()
	self._activeIndex = 0
	self._activeView = nil
	self._keyCommandMap = {}

	self._boundsReachedInputInterceptor = SignalInterceptor.new(
				input.keyPressSignal, 400, self._onBoundsReachedKeyPress, self)
	self._boundsReachedInputInterceptor:deactivate()

	self._lastTimeBumped = 0

	local signal = self:addBubblingSignal("internalFocusGained")
	signal:add(self._onFocusGained, self)

	local returnValue = IteratingView.createResources(self)

	self:_initScrollingAnimation()

	self._behaviour = BehaviourFactory.createBehaviour(
			self:getProperty('behaviour'))

	return returnValue
end

function CarouselView:_initScrollingAnimation()
	local defaultScrollingAnimation = self:getProperty("scrollingAnimation")
	self._scrollingIteratorOptions = IteratorOptions.new({})
	self._listScroller = self:_createDefaultListScroller()
	self:_updateScrollingAnimation(defaultScrollingAnimation)
end

function CarouselView:_updateScrollingAnimation(scrollingAnimation)
	if scrollingAnimation then
		self._preventScrolling = false
		self._listScroller:setAnimation(scrollingAnimation)
	else
		self._preventScrolling = true
	end
end

function CarouselView:_createDefaultListScroller()
	return ListScroller.new(
	{
		keyRepeatInterceptor = KeyRepeatInterceptor.new(300),
		scrollingTimer = ScrollingTimer.new(CarouselView.defaultScrollingInterval),
		callbackContext = self,
		callbacks =
		{
			getDirectionForKey = self._getDirectionForKey,
			hasElementsInDirection = self._hasElementsInDirection,
			scroll = self._tryToScroll
		}
	})
end

function CarouselView:_tryToScroll(direction, scrollingIteratorOptions)
	if not self._preventScrolling then
		self:_shiftPage(direction, interval)
	end
end

function CarouselView:_shiftPage(direction, scrollingIteratorOptions)
	newActiveIndex = self:_getNextActiveIndex(direction)
	local ops = OpsContainer.new()
	-- updateActiveIndex assumes this is a javascript index, so we need to adjust it
	ops:setLocal(self.indexOpsKeyword, newActiveIndex - 1)
	self:_updateActiveIndex(ops, scrollingIteratorOptions)
	self:_requestFocusToActiveView()
end

function CarouselView:_getNextActiveIndex(direction)
	local upperIndexLimit = self:getNumChildElements() - 1
	local lowerIndexLimit = 1
	local rawNewActiveIndex
	local newActiveIndex

	if self:getProperty("iterator"):isForward(direction) then
		rawNewActiveIndex = self._activeIndex + 1
	elseif self:getProperty("iterator"):isBackward(direction) then
		rawNewActiveIndex = self._activeIndex - 1
	end

	if rawNewActiveIndex then
		if self:getProperty("behaviour") == "WrappingBehaviour" then
			newActiveIndex = self:_wrapIndex(rawNewActiveIndex)
		elseif self._activeIndex > lowerIndexLimit and self._activeIndex < upperIndexLimit then
			newActiveIndex = rawNewActiveIndex
		end
	end

	return newActiveIndex
end

function CarouselView:_wrapIndex(index)
	zeroBasedIndex = index - 1
	zeroBasedWrappedIndex = zeroBasedIndex % (self:getNumChildElements() - 1)
	return zeroBasedWrappedIndex + 1
end

function CarouselView:_requestFocusToActiveView()
	signalBus:getSignal("requestFocus"):dispatch(self:_getActiveNode())
end

function CarouselView:_requestFocusOnNodeIndex(nodeIndex)
	local nodeView = self._iterationViews[nodeIndex]
	if nodeView then
		local nodeToFocus = nodeView:getRootNode():getSceneNode()
		signalBus:getSignal("requestFocus"):dispatch(nodeToFocus)
	end
end

function CarouselView:_hasElementsInDirection(direction)
	return self:_getNextActiveIndex(direction) ~= nil
end

function CarouselView:_getActiveNode()
	local activeView = self._iterationViews[self._activeIndex]
	if activeView then
		return activeView:getRootNode():getSceneNode()
	end
end

function CarouselView:getFocusCandidate()
	return self:_getActiveNode()
end

function CarouselView:preflight(ops)
	IteratingView.preflight(self, ops)

	self:_sanitiseActiveIndex()
	self:_updateActiveIndex(ops)

	return true
end

function CarouselView:_onFocusGained(focusedNode)
	local child = self:_iterativelyGetChildNodeWhichContainsNode(focusedNode)
	self:_updateActiveIndexBasedOnNewlyFocusedNode(child)

	if self._listScroller then
		self._listScroller:activate()
	end

	if self._boundsReachedInputInterceptor then
		self._boundsReachedInputInterceptor:activate()
	end
end

function CarouselView:_onFocusLost(nodeWhichLostFocus)
	IteratingView._onFocusLost(self, nodeWhichLostFocus)
	-- We need to deactivate the input interceptor when the list loses focus,
	-- (ie. a descendant loses focus) otherwise when the user press
	-- "left/right/whatever" in another list this one (which lost focus)
	-- will continue to move (as it'll still be intercepting the input)
	if self._listScroller and not self._listScroller:isScrolling() then
		self._listScroller:deactivate()
	end
	
	if self._boundsReachedInputInterceptor then
		self._boundsReachedInputInterceptor:deactivate()
	end
end

function CarouselView:_updateActiveIndexBasedOnNewlyFocusedNode(newlyFocusedNode)
	if newlyFocusedNode then
		local ops = OpsContainer.new()
		ops:setLocal(self.indexOpsKeyword, newlyFocusedNode:getIndexInParent())
		self:_updateActiveIndex(ops)
	end
end

function CarouselView:_updateActiveIndex(ops, scrollingIteratorOptions)
	if #self._iterationViews > 0 then
		local oldActiveIndex = self._activeIndex
		local newActiveIndex = self:_getIndexValueFromOps(ops) or self._activeIndex
		self._activeIndex = math.max(newActiveIndex, 1)
		self._activeView = self._iterationViews[newActiveIndex]

		local adjustedViews, adjustedIndex, delta
				= self._behaviour:adjustElementsPosition(
					self._iterationViews,
					oldActiveIndex,
					newActiveIndex)

		if (delta ~= 0) then
			local startingPositionShift =
					math.min(#adjustedViews, self:_getSelectedPos())

			local options = IteratorOptions.new(
			{
				itemInFirstPosition = adjustedIndex,
				startingPositionShift = startingPositionShift,
				firstPositionOffsetCap = self:getProperty("selectedPosMaxOffset"),
				extraSpacing = {
					[adjustedIndex] = self:getProperty("spacingAfterHighlightedItem")
				},
				positionDelta = delta
			})
			if scrollingIteratorOptions ~= nil then
				options:merge(scrollingIteratorOptions)
			else
				options:addOption("easingFn", self:getProperty("easingFunction"))
				options:addOption("duration", self:getProperty("transitionDuration"))
			end

			self._behaviour:complementIteratorOptions(options)

			self:getProperty("iterator"):iterate(adjustedViews, options)
		end

		-- Let the newly active view know that it is active.
		if self._activeView ~= nil then
			self:_supplyActiveFlag(
					ops, oldActiveIndex, false, OpsContainer.OPERATION_DELETE)
			self:_supplyActiveFlag(
					ops, newActiveIndex, true, OpsContainer.OPERATION_CREATE)
		end
	end
end

function CarouselView:_hasActiveIndexReachedBounds()
	return ((self._activeIndex == 1)
			or (self._activeIndex == #self._iterationViews))
end

function CarouselView:_onBoundsReachedKeyPress(keyCode, keyModifier)
	local direction = self:_getDirectionForKey(keyCode, keyModifier)
	local boundsReachedBehaviourProperty = self:getProperty("boundsReachedBehaviour")

	local result = true

	if boundsReachedBehaviourProperty == "WRAP_FOCUS" then
		result = self:_wrapFocus(direction)
	elseif boundsReachedBehaviourProperty == "BUMP" then
		result = self:_bump(direction)
	end

	return result
end

function CarouselView:_bump(direction)
	local bumpDelta = self:_getBumpDelta(direction)

	if bumpDelta then
		local bumpDuration = self:getProperty("bumpTransitionDuration")
		local time = timer.getTimeInMilliseconds()

		if time - self._lastTimeBumped > bumpDuration then
			NodeBumper.bump(
					self:getRootNode():getSceneNode(),
					self:getProperty("iterator"):getPropertyManager(),
					bumpDelta,
					bumpDuration)

			self._lastTimeBumped = time

			return false
		end
	end
end

function CarouselView:_getBumpDelta(direction)
	if not direction then
		return nil
	end

	local sign = 0
	if self:_isTryingToIteratePastEndOfList(direction) then
		sign = -1
	elseif self:_isTryingToIteratePastBeginningOfList(direction) and
			(self._activeIndex == 1) then
		sign = 1
	else
		return nil
	end

	return sign * self:getProperty("bumpDistance")
end

function CarouselView:_wrapFocus(direction)
	if self:_isTryingToIteratePastEndOfList(direction) then
		self:_requestFocusOnNodeIndex(1)

		return false
	elseif self:_isTryingToIteratePastBeginningOfList(direction) then
		self:_requestFocusOnNodeIndex(#self._iterationViews)

		return false
	end
end

function CarouselView:_postApplyAggregates()
	self:_sanitiseActiveIndex()
end

function CarouselView:_sanitiseActiveIndex()
	local numViews = #self._iterationViews

	if self._activeIndex == 0 then
		if numViews > 0 then
			self._activeIndex = math.min(numViews, self:_getSelectedPos())
		end
	elseif self._activeIndex > numViews then
		self._activeIndex = math.max(numViews, 0)
	end
end

function CarouselView:_iterateElementsOnNodePropertyChange()
	-- We assume here that the first node property change signal
	-- is a result of the dimensions of the list item nodes being set,
	-- so we don't animate to the new positions; any subsequent adjustments
	-- to list item dimensions result in animated iterations.
	if not self._hasInitializedPosition then
		self:_setElementsPosition()
		self._hasInitializedPosition = true
	else
		self:_setElementsPosition(
			IteratorOptions.new({
				easingFn = self:getProperty("easingFunction"),
				duration = self:getProperty("transitionDuration")
			})
		)
	end
end

function CarouselView:_setElementsPosition(iteratorOptionsOverrides)
	if #self._iterationViews > 0 then
		local adjustedViews, adjustedIndex, delta
				= self._behaviour:adjustElementsPosition(
					self._iterationViews,
					self._activeIndex,
					self._activeIndex)

		local startingPositionShift =
				math.min(#adjustedViews, self:_getSelectedPos())

		local options = IteratorOptions.new(
		{
			itemInFirstPosition = adjustedIndex,
			startingPositionShift = startingPositionShift,
			firstPositionOffsetCap = self:getProperty("selectedPosMaxOffset"),
			extraSpacing =
			{
				[adjustedIndex] = self:getProperty("spacingAfterHighlightedItem")
			},
			positionDelta = delta
		})

		if iteratorOptionsOverrides then
			options:merge(iteratorOptionsOverrides)
		end

		self._behaviour:complementIteratorOptions(options)
		self:getProperty("iterator"):iterate(adjustedViews, options)
	end
end

function CarouselView:_getSelectedPos()
	local selectedPos = self:getProperty("selectedPos")
	if self:getProperty("flexibleSelectedPos") then
		selectedPos = math.min(self._activeIndex, selectedPos)
	end

	return selectedPos
end

function CarouselView:_getIndexValueFromOps(ops)
	local op = ops:getLocal(self.indexOpsKeyword)
			or ops:getLocal(self.legacyIndexOpsKeyword)
	if op ~= nil then
		-- Note adding 1 because the supplied index will be zero-based.
		return op.value + 1
	else
		return nil
	end
end

function CarouselView:_indexOfViewInIterationViews(view)
	for index, iterationView in pairs(self._iterationViews) do
		if view == iterationView then
			return index
		end
	end
	return nil
end

function CarouselView:_supplyActiveFlag(ops, index, value, verb)
	-- Since we added the postApplyAggregates function, we can get to this function
	-- with index = 0 for a carousel that is initialised with no elements and later
	-- filled with data that we get from an async request
	if index <= 0 or index > #self._iterationViews then
		return
	end

	local aggregate = ops:getAggregate(index)
	if aggregate == nil then
		aggregate = ops:addAggregate(index)
	end

	local iterationView = self._iterationViews[index]

	-- Must ensure the active flag is around when update() is called
	aggregate:setLocal(self.activeOpsName, value, verb)

	-- Must not preflight with the aggregates passed because the preflight for those
	-- ops was already called. Instead must pass a new container with just the "active"
	-- ops.
	local localOps = OpsContainer.new()
	localOps:setLocal(self.activeOpsName, value, verb)
	iterationView:preflight(localOps)

	-- if it's active add the higlighted tag
	if value then
		iterationView:getRootNode():getSceneNode():addTag(CarouselView.activeItemTag)
	else
		iterationView:getRootNode():getSceneNode():removeTag(CarouselView.activeItemTag)
	end

end

function CarouselView:_validateBoundsReachedBehaviourUsage(newValue)
	if self:getProperty('behaviour') then
		log.error("CarouselView: You should not use behaviour property together with boundReachedBehaviour property")
	end

	if self:getProperty('enableBumpOnBoundsReached') then
		log.error("CarouselView: You should not use enableBumpOnBoundsReached property together with boundReachedBehaviour property")
	end

	if self:_isBoundsReachedBehaviourValueValid(newValue) ~= true then
		log.error("CarouselView: The value provided for the boundsReachedBehaviour property is not valid."..
		"Please chose one of the following: 'nil', 'LOCK_FOCUS', 'WRAP_FOCUS' or 'BUMP'")
	end
end

function CarouselView:_isBoundsReachedBehaviourValueValid(newValue)
	return newValue == nil or
			newValue == CarouselView.boundsReachedBehaviourValues.LOCK_FOCUS or
			newValue == CarouselView.boundsReachedBehaviourValues.WRAP_FOCUS or
			newValue == CarouselView.boundsReachedBehaviourValues.BUMP
end

function CarouselView:getActiveIndex()
	return self._activeIndex
end

function CarouselView:getActiveView()
	return self._activeView
end

function CarouselView:_isTryingToIteratePastEndOfList(direction)
	local iterator = self:getProperty("iterator")
	if iterator:isForward(direction) and (self._activeIndex == #self._iterationViews) then
		return true
	end

	return false
end

function CarouselView:_isTryingToIteratePastBeginningOfList(direction)
	local iterator = self:getProperty("iterator")
	if iterator:isBackward(direction) and (self._activeIndex == 1) then
		return true
	end

	return false
end

function CarouselView:dispose()
	if self._listScroller then
		self._listScroller:dispose()
		self._listScroller = nil
	end
	self._boundsReachedInputInterceptor:dispose()

	return IteratingView.dispose(self)
end

return CarouselView
