-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ViewPropertiesRegistry = plugins.views:lazyRequire("core.definitions.ViewPropertiesRegistry")
local NoIterator = plugins.views:lazyRequire("properties.iterator.NoIterator")

local KEY_DIRECTIONS = {}
if input then
	KEY_DIRECTIONS =
	{
		[input.keyCodes.KEY_UP]    = "up",
		[input.keyCodes.KEY_DOWN]  = "down",
		[input.keyCodes.KEY_LEFT]  = "left",
		[input.keyCodes.KEY_RIGHT] = "right"
	}
end

local ElementView = ViewClass('ElementView')
local AbstractListView = class(ElementView)

AbstractListView.typeName = 'AbstractListView'

AbstractListView.elements =
{
	{
		className = 'ContainerNodeDirective',
		type = ElementView.TYPE_DIRECTIVE,
		parent = nil
	}
}

AbstractListView.propertyDefinitions = ViewPropertiesRegistry.mergePropertyDefinitions(
	ElementView.propertyDefinitions,
	{
		iterator =
		{
			default = NoIterator.new(),
			onChange = function(abstractListView)
				abstractListView:_setElementsPosition()
			end
		},
		-- The 'lockFocus' property specifies if the behaviour when the
		-- user navigates through the list (in the same iterating direction,
		-- i.e. left/right for horizontal iterators, up/down for Vertical)
		-- If set to true, the focus has to stay in the view, otherwise
		-- it's free to go wherever the focus manager decides.
		lockFocus =
		{
			default = false
		},
		-- This property is similar to 'lockFocus', but instead of locking
		-- the focus in the iterating directions, the user is free to
		-- specify the locked directions, e.g. {"left", "up"}
		lockFocusInDirections =
		{
			default = {}
		}
	}
)

function AbstractListView:interceptFocusLoss(oldNode, newNode, direction, callingView)
	local childNode = self:_getChildNodeWhichContainsNode(oldNode, callingView)

	if self:_shouldKeepFocus(childNode, newNode, direction) then
		newNode = oldNode
	end

	if self:hasParent() then
		newNode = self:getParent():interceptFocusLoss(
				oldNode, newNode, direction, self)
	end

	return newNode
end

function AbstractListView:_shouldKeepFocus(previouslyFocusedChild, newFocusedNode, direction)
	if previouslyFocusedChild then
		local newNodeIsDescendant = self:_isDescendant(newFocusedNode)

		if (not newNodeIsDescendant) and self:_shouldLockFocus(
				previouslyFocusedChild, newFocusedNode, direction) then
			return true
		end
	end

	return false
end

function AbstractListView:_shouldLockFocus(oldNode, newNode, direction)
	if self:getProperty("iterator"):isIteratingDirection(direction) then
		if self:getProperty("lockFocus") then
			return true
		end
	else
		for i, lockedDirection in ipairs(self:getProperty("lockFocusInDirections")) do
			if direction == lockedDirection then
				return true
			end
		end
	end

	return false
end

-- This method can be used as an auxiliary function in the "interceptFocus"
-- methods. It returns the direct child of the current view that holds the focus.
-- e.g.
--      1        Supposing 1 is the view calling this method and 4 is the
--     / \       focused node, this function will return the node 3 (i.e.
--    2   3      the callingView)
--        |
--        4
function AbstractListView:_getChildNodeWhichContainsNode(focusedNode, callingView)
	local nodeInView = nil
	local selfNode = self:getRootNode():getSceneNode()
	local callingNode = callingView and callingView:getRootNode():getSceneNode()

	local nodeIsDirectChild = (selfNode:getId() == focusedNode:getParent():getId())
	local nodeIsDescendant =
			callingNode and (selfNode:getId()==callingNode:getParent():getId())

	if nodeIsDirectChild then
		nodeInView = focusedNode
	elseif nodeIsDescendant then
		nodeInView = callingNode
	end

	return nodeInView
end

-- Same as AbstractListView:_getChildNodeWhichContainsNode(), but with O(n)
-- complexity (n = height of the tree) (in case we have no 'callingView'
-- available to optimise the process)
function AbstractListView:_iterativelyGetChildNodeWhichContainsNode(focusedNode)
	local root = self:getRootNode():getSceneNode()
	local child = focusedNode

	while child:hasParent() and child:getParent() ~= root do
		child = child:getParent()
	end

	return child
end

function AbstractListView:_getDirectionForKey(keyCode, keyModifier)
	local iteratingDirections = self:getProperty("iterator"):getIteratingDirections()
	local direction = KEY_DIRECTIONS[keyCode]

	if direction and iteratingDirections then
		local isIteratingDirection = iteratingDirections[direction]
		if isIteratingDirection then
			return direction
		end
	end

	return nil
end

return AbstractListView
