-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

-- Prefix to special keys added by the JsObjectToLuaTableConverter utility.
local IGNITION_KEY_PREFIX = "$IGN_"
local STARTS_WITH_JS2LUA_KEY_PREFIX_PATTERN = "^%" .. IGNITION_KEY_PREFIX
local IS_ARRAY_KEY = IGNITION_KEY_PREFIX .. "IS_ARRAY"
local OBJECT_ID_KEY = IGNITION_KEY_PREFIX .. "ID"
local IMMUTABLE_KEY = IGNITION_KEY_PREFIX .. "IMMUTABLE"
local CONVERTED_TABLES_REGISTRY = IGNITION_KEY_PREFIX .. "CONVERTED_TABLES_REGISTRY"

local RetainedObjectUtils = class(function(self)
end)

RetainedObjectUtils.OBJECT_ID_KEY = OBJECT_ID_KEY

function RetainedObjectUtils.getObjectId(model)
	if (type(model) ~= 'table') then
		error(
			'RetainedObjectUtils.getObjectId: This method only supports ' ..
			'objects at this time. Supplied was ' .. tostring(model));
	end

	return model[OBJECT_ID_KEY]
end

function RetainedObjectUtils.setObjectId(model, id)
	model[OBJECT_ID_KEY] = id
end

function RetainedObjectUtils.getObjectIds(models)
	local ids = {}

	for i = 1, #models do
		table.insert(ids, RetainedObjectUtils.getObjectId(models[i]))
	end

	return ids;
end

function RetainedObjectUtils.isArray(model)
	return model[IS_ARRAY_KEY] == true
end

function RetainedObjectUtils.objectIsImmutable(model)
	return model[IMMUTABLE_KEY] == true
end

function RetainedObjectUtils.isReservedKey(key)
	return string.find(key, STARTS_WITH_JS2LUA_KEY_PREFIX_PATTERN) == 1
end

function RetainedObjectUtils.find(idToFind)
	local found = {}
	RetainedObjectUtils._serialize(
			RetainedObjectUtils._find(idToFind, _G, {}), "_G", found)
	return found
end

function RetainedObjectUtils._find(idToFind, current, visited)
	local found = {elements = {}, children = {}}
	visited[current] = true
	for i, v in pairs(current) do
		if i == OBJECT_ID_KEY and ((idToFind == nil) or (idToFind == v)) then
			table.insert(found.elements, v)

		elseif type(v) == "table" and not visited[v] then
			if i ~= CONVERTED_TABLES_REGISTRY then
				local foundInChild = RetainedObjectUtils._find(idToFind, v, visited)

				local numChildren = 0
				for c, val in pairs(foundInChild.children) do
					numChildren = numChildren + 1
				end

				if #foundInChild.elements ~= 0 or numChildren ~= 0 then
					found.children[i] = foundInChild
				end
			end
		end
	end
	return found
end

function RetainedObjectUtils._serialize(findResult, path, serializedResult)
	for i, v in ipairs(findResult.elements) do
		table.insert(serializedResult, {id = v, path = path})
	end

	for i, v in pairs(findResult.children) do
		RetainedObjectUtils._serialize(v, path .. "." .. tostring(i), serializedResult)
	end
end

return RetainedObjectUtils