-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local BuilderCursor = plugins.views:lazyRequire('utils.BuilderCursor')
local RetainedObjectUtils = plugins.views:lazyRequire('utils.RetainedObjectUtils')
local TypeChecker = plugins.views:lazyRequire('utils.TypeChecker')

local ModelDiffer = class(function(self)
	self._cache = {}
	self._cursor = nil
end)

function ModelDiffer:_diffPrimitive(value, cache, key)
	local builder
	local op

	local oldValue = cache[key]
	if (value ~= oldValue) then
		if (oldValue == nil) then
			-- CREATE
			op = OpsContainer.OPERATION_CREATE
		else
			-- UPDATE
			op = OpsContainer.OPERATION_UPDATE
		end

		-- Write to builder
		builder = self._cursor:getBuilder()
		builder:setLocal(key, op, value)
		if op == OpsContainer.OPERATION_UPDATE then
			builder:extendLocal(key, "oldValue", oldValue)
		end

		-- Sync Cache
		cache[key] = value
	end
end

function ModelDiffer:_getObjectId(subject)
	return RetainedObjectUtils.getObjectId(subject)
end

function ModelDiffer:_getObjectIds(subjects)
	return RetainedObjectUtils.getObjectIds(subjects)
end

local function _arrayContainsValue(theTable, theValue)
	for i = 1, #theTable do
		if theTable[i] == theValue then
			return true
		end
	end

	return false
end

function ModelDiffer:_diffArray(arr, cache, key, deletedObjectIds)
	self:_bailIfVisited(arr)
	self._visited[arr] = true

	local builder
	local cacheParent = cache
	local newCache = {}
	cache = cache[key]

	local isObjectArray = true

	-- Check whether all members of the array are tables
	for i = 1, #arr do
		if type(arr[i]) ~= 'table' then
			isObjectArray = false
			break
		end
	end

	if (isObjectArray == false)	then
		log.warn(
			'Currently, the ModelDiffer only supports arrays of objects. ' ..
			'Arrays of simple values are ignored.')
		self._visited[arr] = false
		return
	end

	local modelIds = self:_getObjectIds(arr)
	local cacheIds = self:_getObjectIds(cache)

	local modelIdsLength = #modelIds
	local cacheIdsLength = #cacheIds

	local modelIdsIndex = 1
	local cacheIdsIndex = 1

	local opIndex = 1

	local deletedIds = {}
	for i = 1, #cacheIds do
		local id = cacheIds[i]
		if not _arrayContainsValue(modelIds, id) then
			deletedIds[id] = true
		end
	end

	local createdIds = {}
	for i = 1, #modelIds do
		local id = modelIds[i]
		if not _arrayContainsValue(cacheIds, id) then
			createdIds[id] = true
		end
	end

	while (modelIdsIndex <= modelIdsLength) or (cacheIdsIndex <= cacheIdsLength) do

		-- Process 'CREATE's
		if  (modelIdsIndex <= modelIdsLength)
		and (createdIds[modelIds[modelIdsIndex]] == true)
		then
			-- Create Cache object
			local newCacheItem = {}
			newCache[modelIdsIndex] = newCacheItem
			RetainedObjectUtils.setObjectId(
					newCacheItem, modelIds[modelIdsIndex])

			local childObject = arr[modelIdsIndex]

			self._cursor:enter(opIndex, childObject)
			builder = self._cursor:getBuilder()
			builder:setType(OpsContainer.OPERATION_CREATE)
			self:_diffObjectKeys(childObject, newCacheItem, true, deletedObjectIds)
			self._cursor:exit()

			modelIdsIndex = modelIdsIndex + 1

		-- Process 'DELETES's
		elseif (cacheIdsIndex <= cacheIdsLength)
		 and   (deletedIds[cacheIds[cacheIdsIndex]] == true)
		then
			self._cursor:enter(opIndex)
			builder = self._cursor:getBuilder()
			builder:setType(OpsContainer.OPERATION_DELETE)
			self._cursor:exit()

			if deletedObjectIds ~= nil then
				table.insert(deletedObjectIds, cacheIds[cacheIdsIndex])
			end

			cacheIdsIndex = cacheIdsIndex + 1

		-- Process Existing Keys
		else
			-- Copy pointer to existing element cache object
			local newCacheItem = cache[cacheIdsIndex]
			newCache[modelIdsIndex] = newCacheItem

			if self:_getObjectId(arr[modelIdsIndex]) ~= self:_getObjectId(newCacheItem) then
				RetainedObjectUtils.setObjectId(
						newCacheItem, modelIds[modelIdsIndex])

				log.warn(
					'ModelDiffer._diffArray: Index re-reordering is tentatively ' ..
					'supported but not tested. It is advised that you do not ' ..
					're-order existing array elements.')
			end

			local childObject = arr[modelIdsIndex]

			self._cursor:enter(opIndex, childObject)
			self:_diffObjectKeys(childObject, newCacheItem, false, deletedObjectIds)
			self._cursor:exit()

			modelIdsIndex = modelIdsIndex + 1
			cacheIdsIndex = cacheIdsIndex + 1
		end

		opIndex = opIndex + 1
	end

	-- Copy in the object ID for this array
	RetainedObjectUtils.setObjectId(
				newCache, RetainedObjectUtils.getObjectId(arr))

	-- Set new cache mapping
	cacheParent[key] = newCache

	self._visited[arr] = false
end

function ModelDiffer:_diffObject(obj, cache, key, deletedObjectIds)
	local builder
	local isNewObject = false

	self._cursor:enter(key, obj)

	-- Check if we're creating a new object
	if (cache[key] == nil) then
		isNewObject = true
		builder = self._cursor:getBuilder()
		builder:setType(OpsContainer.OPERATION_CREATE)
		cache[key] = {}
	end

	if RetainedObjectUtils.isArray(obj) then
		self:_diffArray(obj, cache, key, deletedObjectIds)
	else
		self:_diffObjectKeys(obj, cache[key], isNewObject, deletedObjectIds)
	end

	self._cursor:exit()
end

function ModelDiffer:_diffObjectKeys(obj, cache, isNewObject, deletedObjectIds)
	self:_bailIfVisited(obj)
	self._visited[obj] = true

	-- Don't diff immutable objects (saves iterating over values that
	-- the application knows won't have changed)
	if (not isNewObject and self:_objectIsImmutable(obj)) then
		self._visited[obj] = false
		return
	end

	-- Forward Model > Cache (CREATE/UPDATE)
	for key, value in pairs(obj) do
		if not self:_shouldIgnoreKey(key) then
			local type = type(value)

			if value == nil then
				log.warn('ModelDiffer: nil values aren\'t ' .. 'currently supported.')
			elseif type == 'table' and not self:_isPackedObject(value) then
				self:_diffObject(value, cache, key, deletedObjectIds)
			else
				self:_diffPrimitive(value, cache, key)
			end
		end
	end

	-- Backward Cache > Model (DELETE)
	for key in pairs(cache) do
		if not self:_shouldIgnoreKey(key) then
			local builder

			-- If the object key is nil and there is no
			-- matching getter discovery
			if (obj[key] == nil) then
				-- Keys which previously contained object are deleted by
				-- creating an empty, delete-types ops container
				if (type(cache[key]) == 'table') then
					local deletedObject = cache[key]
					self._cursor:enter(key, deletedObject)
					builder = self._cursor:getBuilder()
					builder:setType(OpsContainer.OPERATION_DELETE)
					self._cursor:exit()

					if deletedObjectIds ~= nil then
						table.insert(deletedObjectIds, self:_getObjectId(deletedObject))
					end
				else
					builder = self._cursor:getBuilder()
					builder:setLocal(key, OpsContainer.OPERATION_DELETE, 0)
				end

				-- Delete Cache
				cache[key] = nil
			end
		end
	end

	-- Copy in the object ID for this object
	RetainedObjectUtils.setObjectId(cache, RetainedObjectUtils.getObjectId(obj))

	self._visited[obj] = false
end

function ModelDiffer:_bailIfVisited(obj)
	if self._visited[obj] == true then
		error('Cyclic model value detected')
	end
end

function ModelDiffer:_shouldIgnoreKey(key)
	return RetainedObjectUtils.isReservedKey(key)
end

-- We want to be able to pass some "packed" values from XML/ViewModels down
-- to directives, e.g. font/background colors ({r, g, b, a}), positions, etc
-- In order to do that, the model differ has to treat these values as if they
-- were primitive types, rather than a table/object. This method helps detecting
-- the supported "special tables" ({r,g,b}, {r, g, b, a}, {x, y, z}, {x, y, z, w})
function ModelDiffer:_isPackedObject(val)
	if type(val) ~= "table" then
		return false
	end

	local isPackedObject = false
	local numKeys = 0
	for k, v in pairs(val) do
		if not self:_shouldIgnoreKey(k) then
			numKeys = numKeys + 1
		end
	end

	if numKeys == 3 then
		isPackedObject = TypeChecker.isXYZ(val) or TypeChecker.isRGB(val)
	elseif numKeys == 4 then
		isPackedObject = TypeChecker.isXYZW(val) or TypeChecker.isRGBA(val)
	end

	return isPackedObject
end

function ModelDiffer:_objectIsImmutable(obj)
	return RetainedObjectUtils.objectIsImmutable(obj)
end

function ModelDiffer:diff(model, deletedObjectIds)
	if (type(model) ~= 'table') then
		error(
			'ModelDiffer.diff: This method takes one ' ..
			'argument which must be a table'
		)
	end

	self._cursor = BuilderCursor.new(model)
	self._visited = {}
	self:_diffObjectKeys(model, self._cache, true, deletedObjectIds)

	return self._cursor:getBuilder():getRootOpsContainer()
end

return ModelDiffer
