-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local OpsBuilder = plugins.views:lazyRequire('utils.OpsBuilder')

local BuilderCursor = class(function(self, rootModel)
	self._builder = OpsBuilder.new(rootModel)
	self._virtualCursorNames = {}
	self._virtualCursorModels = {}
	self._builderCursor = {}
	self._cursorSync = true
end)

function BuilderCursor:enter(name, model)
	table.insert(self._virtualCursorNames, name)
	table.insert(self._virtualCursorModels, model)
	self._cursorSync = false
end

function BuilderCursor:exit()
	table.remove(self._virtualCursorNames)
	table.remove(self._virtualCursorModels)

	if (self._cursorSync) then
		self:_builderExit()
	end
end

function BuilderCursor:_builderEnter(name, model)
	table.insert(self._builderCursor, name)
	self._builder:enterChild(name, model)
end

function BuilderCursor:_builderExit()
	table.remove(self._builderCursor)
	self._builder:exitChild()
end

function BuilderCursor:_syncCursors()
	local index
	local len

	if (self._cursorSync) then
		return
	end

	for index = 1, #self._virtualCursorNames do
		if (self._virtualCursorNames[index] ~= self._builderCursor[index]) then
			-- Unwind the _builderCursor and builder until
			-- we're in a clean state.
			while (#self._builderCursor ~= index - 1) do
				self:_builderExit()
			end

			-- Match the _builderCursor cursor to the _virtualCursorNames
			while (index <= #self._virtualCursorNames) do
				self:_builderEnter(
					self._virtualCursorNames[index],
					self._virtualCursorModels[index])
				index = index + 1
			end

			break
		end
	end

	-- We can sometimes reach this case when we have the virtual cursor position one level
	-- up from the builderCursor position (for instance when diffing an array inside
	-- an object where only position 3/4 was changed)
	-- In this case any subsequent changed items inside the parent object will end up
	-- in the child because the builderCursor stack is inconsistent
	while #self._virtualCursorNames < #self._builderCursor do
		self:_builderExit()
	end

	self._cursorSync = true
end

function BuilderCursor:getBuilder()
	self:_syncCursors()
	return self._builder
end

return BuilderCursor