-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

-- The SpacingCalculator provides simple relative position and element size calculation
-- utility methods to be used in determining both absolute and relative positioning
-- of elements being iterated over.

local SpacingCalculator = class(function(
		self, elements, multiplier, propertyManager, margin)
	self._multiplier = multiplier
	self._propertyManager = propertyManager
	if margin == nil then
		self._marginSize = 0
	else
		self._marginSize = margin
	end
	self._nodes = {}
	for i, elem in pairs(elements) do
		self._nodes[i] = elem:getRootNode():getSceneNode()
	end
end)

-- Spacing calculations can be passed a 'pivotMode' to determine between
-- which bounds of the elements the spacing is calculated between.
--
-- Eg. the below diagram visually depicts the 3 different possible outcomes
-- given different 'pivotMode's for a 'fromIndex' of 1 and 'toIndex' of 2
--
--           "center"
--        <------------>
--
--                 "max"
--              <--------->
--  ___________|     ______|  ___
-- |     1     |    |  2   | | 3 |
-- |___________|    |______| |___|
-- |                |
-- <---------------->
--       "min"
function SpacingCalculator:getSpacingBetweenElements(fromIndex, toIndex, pivotMode)
	local nodeRange = self:_getNodeRange(fromIndex, toIndex)
	return self:_getRelativeSpacingBetweenFirstAndLastNodes(nodeRange, pivotMode)
end

function SpacingCalculator:getHalfElementSpacing(elementIndex)
	return self:_getHalfNodeSpacing(self._nodes[elementIndex])
end

function SpacingCalculator:getFullElementSpacing(elementIndex)
	return self:_getFullNodeSpacing(self._nodes[elementIndex])
end

function SpacingCalculator:getNumberOfElements()
	return #self._nodes
end

function SpacingCalculator:_getRelativeSpacingBetweenFirstAndLastNodes(nodes, pivotMode)
	local spacing = 0

	for i, node in pairs(nodes) do
		local previousNode = nodes[i-1]
		spacing = spacing + self:_getAdjacentNodeSpacing(previousNode, node, pivotMode)
	end

	return spacing
end

function SpacingCalculator:_getNodeRange(from, to)
	local nodeRange = {}

	for i, node in pairs(self._nodes) do
		if i <= to and i >= from then
			nodeRange[i] = node
		end
	end

	return nodeRange
end

function SpacingCalculator:_getAdjacentNodeSpacing(previousNode, currentNode, pivotMode)
	local spacing = 0

	if previousNode ~= nil and currentNode ~= nil then
		if pivotMode == "min" then
			spacing = self:_getFullNodeSpacing(previousNode) + self:_getMargin()
		elseif pivotMode == "max" then
			spacing = self:_getFullNodeSpacing(currentNode) + self:_getMargin()
		else
			spacing = self:_getHalfNodeSpacing(previousNode) +
				self:_getHalfNodeSpacing(currentNode) + self:_getMargin()
		end
	end

	return spacing
end

function SpacingCalculator:_getMargin()
	return self._marginSize * self._multiplier
end

function SpacingCalculator:_getHalfNodeSpacing(node)
	return self._propertyManager:getNodeSize(node) / 2 * self._multiplier
end

function SpacingCalculator:_getFullNodeSpacing(node)
	return self._propertyManager:getNodeSize(node) * self._multiplier
end

return SpacingCalculator
