-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local DeltaCalculator = class(function(
		self, spacingCalculator, alignment, iteratorOptions)
	self._spacingCalculator = spacingCalculator

	alignment = alignment or "center"
	self:_checkAlignmentValue(alignment)
	self._alignment = alignment

	self._iteratorOptions = iteratorOptions
end)

local validAlignments = {
	min = true,
	center = true,
	max = true
}

function DeltaCalculator:_checkAlignmentValue(alignment)
	assert(validAlignments[alignment] == true)
end

function DeltaCalculator:getDelta()
	local options = self._iteratorOptions

	self:_checkIndex(options.startingPositionShift)
	self:_checkIndex(options.itemInFirstPosition)

	local delta = self:_getOffsetRequiredToAlignElement()
		+ self:_calculateStartingPositionOffset(options.startingPositionShift)
		+ self:_calculateItemInFirstPositionOffset(options.itemInFirstPosition)

	return delta
end

function DeltaCalculator:_checkIndex(index)
	local numElements = self._spacingCalculator:getNumberOfElements()
	if index <= 0 then
		error(string.format("DeltaCalculator: index should be greater "
				.. "than 0! index = %d", index))
	elseif index > numElements then
		error(string.format("DeltaCalculator: index out of bounds! "
				.. "index = %d, number of elements: %d", index, numElements))
	end
end

function DeltaCalculator:_getOffsetRequiredToAlignElement()
	if self._alignment == "min" then
		return self._spacingCalculator:getHalfElementSpacing(1)
	elseif self._alignment == "max" then
		return -self._spacingCalculator:getHalfElementSpacing(1)
	else
		return 0
	end
end

function DeltaCalculator:_calculateStartingPositionOffset(
		startingPositionShiftIndex)
	-- Offset due to 'startingPositionShift' option is positive as we are moving forwards
	-- to that item's starting position
	return self._spacingCalculator:getSpacingBetweenElements(
			1, startingPositionShiftIndex, self._alignment)
end

function DeltaCalculator:_calculateItemInFirstPositionOffset(
		itemInFirstPositionIndex)
	-- Offset due to the 'itemInFirstPosition' option is negative as we are shifting
	-- items before the item in the first position backwards
	return -self._spacingCalculator:getSpacingBetweenElements(
			1, itemInFirstPositionIndex, self._alignment)
end

return DeltaCalculator
