-- Copyright 2013-2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local OpConversion = {}

-- This is a special case which supports the model forwarding syntax @{} that
-- can be used in view XML, for example:
--
--     <view class="MyView" data="@{}"/>
--
-- In these cases the key in the routing table will just be an empty string.
-- On encountering one of these keys, opsToAggregates() will forward the entire
-- model object down to the child view, rather than an individual property from
-- it.
local MODEL_FORWARDING_KEY = ''

--[[
-- Builds aggregate OpsContainers based on the routing table and current ops.
--]]
function OpConversion.opsToAggregates(ops, routing)

	local aggregatesByElementIndex = {}

	local function getCreateAggregate(elementIndex)
		local aggregate = aggregatesByElementIndex[elementIndex]

		if aggregate == nil then
			aggregate = ops:addAggregate(elementIndex)
			aggregatesByElementIndex[elementIndex] = aggregate
		end

		return aggregate
	end

	local function processTier(ops, routing)

		local function buildAggregate(name, localOrChildOp, handler)
			local route = routing[name]

			if (route ~= nil) then
				-- If the route specifies that this op should be aggregated,
				-- clone it into an aggregate ops container for each element
				-- it should be aggregated to.
				if(route.aggregate ~= nil) then
					for i = 1, #route.aggregate do
						local aggregateInfo = route.aggregate[i]
						local aggregateSlot = aggregateInfo.slot or name
						local aggregateOps  = getCreateAggregate(aggregateInfo.element)

						handler(aggregateOps, aggregateSlot, localOrChildOp)
					end
				end

				-- If the route contains nesting, recurse down to the next tier.
				if (route.nested ~= nil) then
					processTier(localOrChildOp, route.nested)
				end
			end
		end

		local function iterateOps(localOrChildOps, handler)
			for name, localOrChildOp in pairs(localOrChildOps) do
				buildAggregate(name, localOrChildOp, handler)
			end
		end

		iterateOps(ops:getLocals(), function (aggregateOps, aggregateSlot, localOp)
			aggregateOps:cloneToLocal(aggregateSlot, localOp)
		end)

		iterateOps(ops:getChildren(), function (aggregateOps, aggregateSlot, childOp)
			aggregateOps:cloneToChild(aggregateSlot, childOp)
		end)

		buildAggregate(MODEL_FORWARDING_KEY, ops, function (aggregateOps, aggregateSlot, childOp)
			aggregateOps:cloneToChild(aggregateSlot, childOp)
		end)

	end

	processTier(ops, routing)
end

local function _localOpsToRoutingKeys(ops, routing, keyCallback)

	-- Call the callback for each local op
	for name, localOp in pairs(ops:getLocals()) do
		local route = routing[name]

		if (route ~= nil) and (route.push ~= nil) then
			for i = 1, #route.push do
				keyCallback(localOp, route.push[i])
			end
		end
	end
end

local function _opsToRoutingKeys(ops, routing, keyCallback)
	-- Consume Local Props
	_localOpsToRoutingKeys(ops, routing, keyCallback)

	-- Consume Nested props
	for name, childOps in pairs(ops:getChildren()) do
		local route = routing[name]

		if (route ~= nil) and (route.nested ~= nil) then
			_opsToRoutingKeys(childOps, route.nested, keyCallback)
		end
	end
end

--[[
-- Builds an element->properties table based on the ops container provided and
-- the view's routing table.
--]]
function OpConversion.opsToProps(ops, routing)
	assert(ops ~= nil, 'opsToProps: ops must not be nil')
	assert(routing ~= nil, 'opsToProps: routing must not be nil')

	local elementProps = {}

	_opsToRoutingKeys(ops, routing, function (localOp, route)
		local props = elementProps[route.element]

		if props == nil then
			props = {}
			elementProps[route.element] = props
		end

		local prop =
		{
			oldValue = localOp.oldValue,
			value = localOp.value,
			type = localOp.type
		}

		-- Pre-process the value if a preprocessor is set
        -- Is this value a string? Lookup into the preprocessor library
        -- otherwise just invoke the function
        if route.preprocess then
            if type(route.preprocess) == 'table' then
                func = getPreprocessFunction(route.preprocess.func)
                if func then
                    if route.preprocess['args'] then
                        func(prop, unpack(route.preprocess.args))
                    else
                        func(prop)
                    end
                end
            else
		    	route.preprocess(prop)
            end
		end

		if prop.type ~= OpsContainer.OPERATION_IGNORE then
			props[#props + 1] =
			{
				name = route.slot,
				value = prop
			}
		end
	end)

	return elementProps
end

return OpConversion
