-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local colors = plugins.lua:lazyRequire("ansicolors.ansicolors")

local TableHelpers = plugins.views:lazyRequire("core.utils.TableHelpers")
local SignalInfo = plugins.views:lazyRequire("core.signals.SignalInfo")
local Signal = plugins.lua:lazyRequire("signals.Signal")
local ViewPropertiesRegistry = plugins.views:lazyRequire("core.definitions.ViewPropertiesRegistry")
local ViewPropertiesMixin = plugins.views:lazyRequire("utils.ViewPropertiesMixin")
local ErrorContextFinder = plugins.views:lazyRequire("utils.ErrorContextFinder")

local IElement = plugins.views:lazyRequire("core.types.IElement")
local IView = class(IElement, function(self, handle)
	IElement._ctor(self)

	ViewPropertiesMixin._ctor(self, self.getClass().propertyDefinitions)

	self._definitionScope = nil
	self._signals = {}
	self._bubblingSignals = {}
	self._signalRedirects = {}
	self._properties = {}
	self:setHandle(handle)
	self:_register()
	self._focusLayer = self:getProperty("focusLayer")
end)

IView:mixin(ViewPropertiesMixin)

function IView:_register()
	context:getViewRegistry():registerViewInstance(self)
end

function IView:_unregister()
	context:getViewRegistry():unregisterViewInstance(self)
end

--[[
-- @inheritdoc
--
--]]
IView.typeName = 'IView'

--[[
-- Flags that this class and its descendants are view definitions, for code
-- that needs to assert this.
--
--]]
IView.isViewDefinition = true

--[[
-- Structure defining the views dependencies
--
--]]
IView.depends = {
	views = {},
	directives = {}
}

IView.propertyDefinitions =
{
	name =
	{
		default = "",
		onChange = function(view, oldValue, newValue)
			view:setName(newValue)
		end
	},
	focusLayer =
	{
		default = 0,
		onChange = function(view, oldValue, newValue)
			view:_updateFocusLayerProperty()
		end
	}
}

--[[
-- Returns the view's dependencies
--
--]]
function IView:getDependencies()
	local dependsMap = {
		views = {},
		directives = {}
	}

	-- Grab dependencies for self
	TableHelpers.mergeSubKeys(self.depends, dependsMap)

	-- If this is the main view from an XML file, pull in any dependencies
	-- belonging to fragments defined in the same file
	if self._type == IElement.TYPE_VIEW then
		for name, fragment in pairs(self._definitionScope:getFragmentDefinitions()) do
			TableHelpers.mergeSubKeys(fragment:getDependencies(), dependsMap)
		end
	end

	-- Converts the subkeys to array
	for key, value in pairs(dependsMap) do
		dependsMap[key] = TableHelpers.getKeys(value)
	end

	return dependsMap
end

function IView:getDefinitionScope()
	return self._definitionScope
end

function IView:setDefinitionScope(scope)
	self._definitionScope = scope
end

--[[
-- Gives the view the opportunity to make sure it has all the resources it needs
-- in the update pass.
--
--]]
function IView:preflight(ops)
	return true
end

--[[
-- Updates the view based on the operations provided
--
--]]
function IView:update(ops)
	return true
end

function IView:onMediatorReady()
end

function IView:onNodePropertyChange(propertiesChanged)
	local parent = self:getParent()
	if parent then
		parent:onNodePropertyChange(propertiesChanged)
	end
end

--[[
-- Gets the type name of this view
--
--]]
function IView:getTypeName()
	return self.typeName
end

--[[
-- Gets the string name associated with this view, if one has been set
--
--]]
function IView:getName()
	return self._name
end

--[[
-- Sets the string name associated with this view
--
--]]
function IView:setName(name)
	self._name = name
end

--[[
-- Gets the C++ view handle related to this view
--
--]]
function IView:getHandle()
	return self._handle
end

--[[
-- Sets the C++ view handle related to this view
--
--]]
function IView:setHandle(handle)
	self._handle = handle
end

--[[
-- Retrieves the Mediation Group from the ViewHandle if there is a handle
-- present on the view.
--
--]]
function IView:getMediationGroup()
	local handle = self:getHandle()
	if handle ~= nil and handle.getMediationGroup then
		return handle:getMediationGroup()
	end

	return nil
end

--[[
-- Flags that the supplied signal identifier is interesting to JavaScript and
-- so should be forwarded to a JS signal each time it is fired
--
--]]
function IView:registerJsSignalInterest(signalDescriptor)
	if not self:_signalInterestHasBeenRegistered(signalDescriptor) then
		self:_registerSignalInterest(signalDescriptor)

		local signalInfo = SignalInfo.fromDescriptor(signalDescriptor)
		self:_addForwardingSignalListener(signalInfo)
	end

	return true
end

function IView:registerJsBubblingSignalInterest(signalDescriptor)
	local signalInfo = SignalInfo.fromDescriptor(signalDescriptor)
	self:_addBubblingSignalListener(signalInfo)
end

function IView:_signalInterestHasBeenRegistered(signalDescriptor)
	self._interestingSignalDescriptors = self._interestingSignalDescriptors or {}
	return self._interestingSignalDescriptors[signalDescriptor] == true
end

function IView:_registerSignalInterest(signalDescriptor)
	self._interestingSignalDescriptors[signalDescriptor] = true
end

function IView:_addForwardingSignalListener(signalInfo)
	local element = self
	if signalInfo:getSourceName() then
		element = self:getChildElementByName(signalInfo:getSourceName())
	end

	if element == nil then
		self:_error(
			'Could not add signal listener: no element exists with ' ..
			'name ' .. signalInfo:getSourceName()
		)
	end

	local signal = element:getOrCreateSignal(signalInfo:getSignalName())

	signal:add(self:_createSignalHandlingFunction(signalInfo:getDescriptor()))
end

function IView:_addBubblingSignalListener(signalInfo)
	if signalInfo:getSourceName() then
		self:_error(
				'BubblingSignal do not require a source name. Specify ' ..
				'just the signal name.')
	end

	local signal = self:getBubblingSignal(signalInfo:getSignalName())

	if not signal then
		signal = self:addBubblingSignal(signalInfo:getSignalName())
		signal:add(self:_createSignalHandlingFunction(signalInfo:getSignalName()))
	end
end

function IView:_createSignalHandlingFunction(signalDescriptor)
	return function(...)
		local luaArg = {...}
		if (#luaArg == 0) then
			-- This is needed because when dispatch is called without any arguments
			-- ... is an empty array, so dispatchJsSignal will complain that it was
			-- called with an insufficient number of arguments
			self._handle:dispatchJsSignal(signalDescriptor, nil)
		else
			self._handle:dispatchJsSignal(signalDescriptor, ...)
		end
	end
end

function IView:addBubblingSignal(signalName)
	local signal = self._bubblingSignals[signalName]
	if not signal then
		signal = Signal.new()
		self._bubblingSignals[signalName] = signal
	end

	self:_addSignalRedirectsRecursivelyDownwards(
			signalName,
			self._bubblingSignals[signalName])

	return signal
end

function IView:getBubblingSignal(signalName)
	return self._bubblingSignals and self._bubblingSignals[signalName]
end

function IView:_addSignalRedirectsRecursivelyDownwards(signalName, signal)
	local function propagateSignalRedirects(node, sigName, sig)
		for index = 1, #node._childElements do
			if node._childElements[index].getSignal then
				local childSig = node._childElements[index]:getSignal(sigName)
				node:_addSignalRedirect(childSig, sig)
			end
			propagateSignalRedirects(node._childElements[index], sigName, sig)
		end
	end

	propagateSignalRedirects(self, signalName, signal)
end

function IView:_addSignalRedirectsRecursivelyUpwards(signalName, signal)
	local parent = self
	while parent:hasParent() do
		parent = parent:getParent()
		local bubblingSig = parent._bubblingSignals[signalName]
		self:_addSignalRedirect(signal, bubblingSig)
	end
end

function IView:_addSignalRedirect(origin, destination)
	if origin and destination then
		if not self:_alreadyRedirected(origin, destination) then
			local handler = function(...)
				destination:dispatch(...)
			end
			origin:add(handler)

			self:_addRedirectMapping(origin, destination, handler)
		end
	end
end

function IView:_alreadyRedirected(origin, destination)
	if self._signalRedirects[origin] then
		return self._signalRedirects[origin][destination] ~= nil
	end
	return false
end

function IView:_addRedirectMapping(origin, destination, handler)
	local redirectsOrigin = self._signalRedirects[origin]
	if redirectsOrigin then
		self._signalRedirects[origin][destination] = handler
	else
		self._signalRedirects[origin] = {
			[destination] = handler
		}
	end
end

function IView:_removeRedirectionsForAllDestinations()
	for origin, redirects in pairs(self._signalRedirects) do
		for destination, handler in pairs(redirects) do
			origin:remove(handler)
		end
	end
end

--[[
-- Returns the signal with the supplied name, or nil if none exists
--
--]]
function IView:getSignal(name)
	return self._signals[name]
end

--[[
-- Returns the signal with the supplied name, creates if non-existant
--
--]]
function IView:getOrCreateSignal(name)
	local signal = self:getSignal(name)

	if signal == nil then
		signal = self:addSignal(name)
	end

	return signal
end

--[[
-- Adds a signal to the view's signal table, using the supplied name.
--
--]]
function IView:addSignal(name)
	if self._signals[name] ~= nil then
		self:_error('A signal already exists with the name ' .. name)
	else
		local signal = Signal.new()
		self._signals[name] = signal
		self:_addSignalRedirectsRecursivelyUpwards(name, signal)
		return signal
	end
end

--[[
-- Returns a handle to the child view with the supplied name, or nil if none
-- exists.
--
--]]
function IView:getChildViewHandleByName(name)
	for i = 1, self:getNumChildElements() do
		local elem = self:getChildElement(i)

		if elem.getClass().isViewDefinition then
			if elem:getName() == name then
				return elem:getHandle()
			else
				local childViewHandle = elem:getChildViewHandleByName(name)
				if childViewHandle then
					return childViewHandle
				end
			end
		end
	end
	return nil
end

function IView:getDescendantViewHandleById(id)
	for i = 1, self:getNumChildElements() do
		local elem = self:getChildElement(i)

		if elem and elem.getClass().isViewDefinition then
			if elem:getHandle() and elem:getHandle():getUuid() == id then
				return elem:getHandle()
			else
				local childViewHandle = elem:getDescendantViewHandleById(id)
				if childViewHandle then
					return childViewHandle
				end
			end
		end
	end
	return nil
end

function IView:addChildView(childViewHandle)
	if not childViewHandle then
		self:_error("IView:addChildView(childViewHandle): childViewHandle must "
				.. "not be nil!")
	end

	local childView = context:getViewRegistry():getViewInstanceByUuid(
			childViewHandle:getUuid())

	if childView then
		local numChildren = self:getNumChildElements()
		self:setChildElement(numChildren+1, childView)
		local rootNode = self:getRootNode()
		if rootNode then
			context:getSceneNodeOpsQueue():queueOp(
				rootNode:getSceneNode(),
				"addChild",
				childView:getRootNode():getSceneNode())
		end
	else
		self:_error("IView:addChildView(childViewHandle): view instance "
				.. "does not exist!")
	end
end

function IView:removeChildView(childViewHandle)
	if not childViewHandle then
		self:_error("IView:removeChildView(childViewHandle): childViewHandle must "
				.. "not be nil!")
	end

	local childView = context:getViewRegistry():getViewInstanceByUuid(
			childViewHandle:getUuid())

	if childView then
		local removed = self:removeChildElementByUuid(childViewHandle:getUuid())

		if removed then
			local rootNode = self:getRootNode()
			if rootNode then
				context:getSceneNodeOpsQueue():queueOp(
					rootNode:getSceneNode(),
					"removeChild",
					childView:getRootNode():getSceneNode())
			end
		end
	end
end

function IView:getFragmentDefinition(className)
	return self._definitionScope:getFragmentDefinition(className)
end

function IView:getFocusLayer()
	return self._focusLayer
end

function IView:hasFocus()
	return false
end

function IView:dispose()
	local handle = self:getHandle()
	if handle ~= nil and handle.markAsDisposing ~= nil then
		handle:markAsDisposing()
	end

	self:_unregister()
	self:_removeRedirectionsForAllDestinations()
	self:_disposeSignals(self._signals)
	self._signals = {}
	self:_disposeSignals(self._bubblingSignals)
	self._bubblingSignals = {}

	return IElement.dispose(self)
end

function IView:_disposeSignals(signalsTable)
	for name, signal in pairs(signalsTable) do
		signal:dispose()
	end
end

function IView:_updateFocusLayerProperty()
	local focusLayer = self:getProperty("focusLayer")
	if (focusLayer == 0) and self:hasParent() then
		self._focusLayer = self:getParent():getFocusLayer()
	else
		self._focusLayer = focusLayer
	end

	for i = 1, self:getNumChildElements() do
		local child = self:getChildElement(i)
		if child and child.getClass().isViewDefinition then
			child:_updateFocusLayerProperty()
		end
	end
end

function IView:_error(message, elementOrRoutingInfo, erroneousName)
	local prefix = 'Thrown by ' .. self.typeName .. ', UUID ' .. self._uuid .. ': '
	local stacktrace = debug.traceback('', 2)
	local coloredMessage = colors('%{bright red}' .. message)
	local context = ErrorContextFinder.attemptToRetrieveXmlContext(
			self._definitionScope,
			elementOrRoutingInfo,
			erroneousName)

	local fullMessage =
		'\n\n' .. prefix ..
		'\n\n\t' .. coloredMessage .. context ..
		'\n' .. stacktrace

	-- The '2' as the second argument means that when the error is picked up
	-- by lua_pcall, it'll be displays as though the method that called self:_error
	-- was at the top of the stack as opposed to _error itself, which makes it
	-- clearer as to where the actual error happened.
	error(fullMessage, 2)
end

return IView
