
-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local IElement = class(function(self)
	self._uuid = uuid.generateNumber()
	self._rootNode = nil
	self._parent = nil
	self._type = nil
	self._childElements = {}
	self._watchers = {}
end)

--[[
-- Enumeration describing basic view element types
--
--]]
IElement.TYPE_DIRECTIVE = 1;
IElement.TYPE_VIEW = 2;
IElement.TYPE_VIEW_FRAGMENT = 3;

--[[
-- Inheriting views are advised to override this value with whatever may be
-- sensible for that given call. This is primarily used for debug
--
--]]
IElement.typeName = 'IElement'

--[[
-- Returns IElement.TYPE_DIRECTIVE, IElement.TYPE_VIEW or IElement.TYPE_VIEW_FRAGMENT
-- based on how this element was defined.
--
--]]
function IElement:getType()
	return self._type
end

--[[
-- Called by DefinitionScope to flag whether this element is a view or fragment.
--
--]]
function IElement:setType(type)
	self._type = type
end

--[[
-- Gives the Element a chance to the pre-allocate any resources it will
-- need to carry out its setProperties operation
--
--]]
function IElement:createResources()
	return true
end

--[[
-- Disposes of the views resources
--
--]]
function IElement:dispose()
	self._watchers = {}
	return true
end

--[[
-- Returns the View's Root Node
--
--]]
function IElement:getRootNode()
	return nil
end

--[[
-- Returns the View's UUID
--
--]]
function IElement:getUuid()
	return self._uuid
end

function IElement:setParent(parent)
	self._parent = parent
end

function IElement:hasParent()
	return self._parent ~= nil
end

function IElement:getParent()
	return self._parent
end

function IElement:hasChildElement(index)
	return self._childElements[index] ~= nil
end

function IElement:getChildElement(index)
	return self._childElements[index]
end

function IElement:getNumChildElements()
	if not self._childElements then
		return 0
	end
	return #self._childElements
end

function IElement:getChildElementOrDie(index)
	local element = self._childElements[index]

	if element == nil then
		self:_error('No child element exists at index ' .. index)
	end

	return element
end

function IElement:getChildElementByName(name)
	for index = 1, #self.elements do
		if self.elements[index].name == name then
			return self:getChildElement(index)
		end
	end
end

function IElement:setChildElement(index, view)
	self._childElements[index] = view
	if (view ~= nil) then
		view:setParent(self)
	end
end

function IElement:removeChildElementByUuid(uuid)
	local numChildElements = self:getNumChildElements()
	for i=1, numChildElements do
		local elem = self:getChildElement(i)
		if elem:getUuid() == uuid then
			table.remove(self._childElements, i)
			return true
		end
	end

	return false
end

function IElement:getFocusForwardedNode()
	return nil
end

function IElement:interceptFocusLoss(oldNode, newNode, direction, callingView)
	if self:hasParent() then
		newNode = self:getParent():interceptFocusLoss(
				oldNode, newNode, direction, self)
	end
	return newNode
end

function IElement:interceptFocusGain(oldNode, newNode, direction, callingView)
	if self:hasParent() then
		newNode = self:getParent():interceptFocusGain(
				oldNode, newNode, direction, self)
	end
	return newNode
end

function IElement:addWatcher(property, watcher)
	if not self._watchers[property] then
		self._watchers[property] = {}
	end
	table.insert(self._watchers[property], watcher)
end

function IElement:appendWatchers(property, watchers)
	if watchers ~= nil then
		for watcher, _ in pairs(watchers) do
			self:addWatcher(property, watcher)
		end
	end
end

function IElement:hasWatchers(property)
	return (self._watchers[property] ~= nil)
end

function IElement:getWatchers(property)
	return self._watchers[property]
end

return IElement
