-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local DefaultPropertyFactory = plugins.views:lazyRequire('properties.DefaultPropertyFactory')

local ViewPropertiesRegistry = class(function(self, propertyDefinitions)
	self._properties = {}

	for propertyName, propertyDefinition in pairs(propertyDefinitions) do
		self:_addProperty(propertyName, propertyDefinition)
	end
end)

ViewPropertiesRegistry._propertyFactoryCache = {}

function ViewPropertiesRegistry.mergePropertyDefinitions(base, propertyDefinitions)
	-- Overrides/Creates specific keys provided
	for propertyName, property in pairs(base) do
		if not propertyDefinitions[propertyName] then
			propertyDefinitions[propertyName] = property
		end
	end

	return propertyDefinitions
end

function ViewPropertiesRegistry:setView(view)
	self._view = view
end

function ViewPropertiesRegistry:_addProperty(name, propertyDefinition)
	if propertyDefinition.immutable == true and propertyDefinition.onChange then
		log.warn("Property with name '" .. name .. "' has been defined as immutable " ..
			"but also provided with an 'onChange' function, which will never be called")
	end

	self._properties[name] =
	{
		onChange = propertyDefinition.onChange,
		immutable = propertyDefinition.immutable or false,
		required = propertyDefinition.required
	}

	if propertyDefinition.default then
		self:_updatePropertyFromDescription(name,	propertyDefinition.default)
	end
end

function ViewPropertiesRegistry:_updatePropertyFromDescription(name, description)
	local property = self._properties[name]
	property.description = description
	property.value = _getPropertyFactory(name):create(name, description)
end

function _getPropertyFactory(propertyName)
	local propertyFactory = ViewPropertiesRegistry._propertyFactoryCache[propertyName]
	if propertyFactory == nil then

		local function firstToUpper(str)
			return (str:gsub("^%l", string.upper))
		end

		local function _require(fullyQualifiedClassName)
			propertyFactory = lazyRequire(fullyQualifiedClassName)
		end

		local propertyFactoryClassPath = "properties." .. firstToUpper(propertyName) .. "Factory"
		local success, err = pcall(_require, propertyFactoryClassPath)
	

		if not success then
			propertyFactory = DefaultPropertyFactory
		end

		ViewPropertiesRegistry._propertyFactoryCache[propertyName] = propertyFactory
	end

	return propertyFactory.new()
end

function ViewPropertiesRegistry:accepts(propertyName)
	return self._properties[propertyName] ~= nil
end

function ViewPropertiesRegistry:get(propertyName)
	return self._properties[propertyName].value
end

function ViewPropertiesRegistry:set(propertyName, newDescription)
	local property = self._properties[propertyName]
	if not self:accepts(propertyName) then
		error("View '" .. self._view.typeName .. "' does not accept " ..
				"property '" .. propertyName .. "'.")
	elseif property.immutable == true and property.value then
		error("Property with name '" .. propertyName .. "' is immutable - refusing to " ..
			"change value")
	else
		local oldValue = property.value
		self:_updatePropertyFromDescription(propertyName, newDescription)
		self:_triggerPropertyChangeHandler(propertyName, oldValue, property.value)
	end
end

function ViewPropertiesRegistry:validateProperties()
	for propName, property in pairs(self._properties) do
		if property.required and not property.value then
			error("Property with name '" .. propName .. "' is required and has not been set")
		end
	end
end

function ViewPropertiesRegistry:getProperties()
	local propertyValues = {}
	for key, property in pairs(self._properties) do
		propertyValues[key] = property.value
	end
	return propertyValues
end

function ViewPropertiesRegistry:_triggerPropertyChangeHandler(
		propertyName, oldValue, newValue)
	if not self._properties[propertyName]
			or not self._properties[propertyName].onChange
			or (oldValue == newValue) then
		return nil
	end
	return self._properties[propertyName].onChange(
			self._view, oldValue, newValue)
end

return ViewPropertiesRegistry
