-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ffi = lazyRequire('ffi')
local stringx = plugins.lua:lazyRequire('pl.stringx')
local plUtils = plugins.views:lazyRequire('pl.utils')

local FileLoader = plugins.views:lazyRequire('utils.FileLoader')

--[[
-- Constructor for the Definition table. This prototype is
-- responsible for loading and unloading definitions (modules)
-- from a given search path
--
--]]
local DefinitionTable = class(function (inst, validatorFn, searchPrefix)
	-- Reference to the validation function
	inst._validatorFn = validatorFn or function () return true end

	-- Search path string from which lua definitions are loaded
	inst._luaSearchPath = ''

	-- Search path string from which xml definitions are loaded
	inst._xmlSearchPath = ''

	-- Internal registry
	inst._defs = {}

	-- Search prefix prepended to definition names when loading
	inst._searchPrefix = searchPrefix or ""

	inst._pathPendingRevert = false
end)

local _dumpGeneratedLuaFilesOnError = false

function DefinitionTable.setDumpGeneratedLuaFilesOnError(value)
	_dumpGeneratedLuaFilesOnError = value
end

--[[
-- Sets the package search path used when including definitions;
-- expects directory path list of the form /foo/dir/;/bar/dir/
--
--]]
function DefinitionTable:setLuaSearchPath(path)
	self._luaSearchPath = self:_mapEachPath(path, function(path)
			return plugins.resolvePath(path) or path
	end)
end

function DefinitionTable:_mapEachPath(semicolonSeparatedPaths, mapFn)
	-- The path variable can actually contain several semicolon-separated paths, so
	-- we have to split it and resolve each one separately.
	local paths = stringx.split(semicolonSeparatedPaths, ';')
	local mappedPaths = ''

	for i, v in ipairs(paths) do
		if i > 1 then
			mappedPaths = mappedPaths .. ';'
		end

		mappedPaths = mappedPaths .. mapFn(paths[i])
	end

	return mappedPaths
end

--[[
--	Gets the package search path used when including definitions
--
--]]
function DefinitionTable:getLuaSearchPath()
	return self._luaSearchPath
end

--[[
-- Sets the package search path used when including definitions;
-- expects directory path list of the form /foo/dir/;/bar/dir/
--
--]]
function DefinitionTable:setXmlSearchPath(path)
	self._xmlSearchPath = self:_mapEachPath(path, function(path)
			return plugins.resolvePath(path) or path
	end)
end

--[[
--	Gets the package search path used when including definitions
--
--]]
function DefinitionTable:getXmlSearchPath()
	return self._xmlSearchPath
end

--[[
-- Adds the name prefix to the package name
--
--]]
function DefinitionTable:_addNamePrefix(className)
	if string.len(self._searchPrefix) > 0 then
		return self._searchPrefix .. className
	else
		return className
	end
end

--[[
-- Check to see if a given definition (module) is available
--
--]]
function DefinitionTable:_isDefAvailable(className)
	-- Check to see if the module (function) has already been loaded into memory
	if package.loaded[className] then
		return true
	else
		-- Otherwise, build a searcher to check for existence
		for _, searcher in ipairs(package.searchers or package.loaders) do
			local searchFn = searcher(className)
			if type(searchFn) == 'function' then
				package.preload[className] = searchFn
				return true
			end
		end
		return false
	end
end

--[[
-- Loads a given definition from the Lua search path my className/key
--
--]]
function DefinitionTable:loadFromLuaSearchPath(className)
	local retVal
	local nameToRequire
	local prefixedName = self:_addNamePrefix(className)
	local pathRevertOwner = false

	-- Temporarily set the include path to the search path
	local sysPath = package.path

	if self._pathPendingRevert == false then
		package.path = self:_getRequireFormattedLuaSearchPath() .. ';' .. sysPath
		self._pathPendingRevert = true
		pathRevertOwner = true
	end

	if self:has(className) then
		log.error('DefinitionTable:loadFromLuaSearchPath(className) already ' ..
			'loaded definition with className ', className)
	else
		if self:_isDefAvailable(prefixedName) then
			nameToRequire = prefixedName
		elseif self:_isDefAvailable(className) then
			nameToRequire = className
		end

		if nameToRequire ~= nil then
			retVal = self:_requireClass(className, nameToRequire)
		else
			log.error('DefinitionTable:loadFromLuaSearchPath(className) ',
				'with className', className, 'could not be found.')
		end
	end

	-- Revert the search path
	if self._pathPendingRevert == true and pathRevertOwner == true then
		package.path = sysPath
		self._pathPendingRevert = false
	end

	return retVal
end

function DefinitionTable:_getRequireFormattedLuaSearchPath()
	-- We need to convert directory paths such as /foo/bar/ to the form
	-- /foo/bar/?.lua expected by lua's 'require' function
	return self:_mapEachPath(self:getLuaSearchPath(), function(path)
			return path .. '?.lua'
	end)
end

function DefinitionTable:_requireClass(className, fullyQualifiedClassName)
		local def

		-- PCall wrapper for invoking require with error checking
		local function requireWrapper(fullyQualifiedClassName)
			def = lazyRequire(fullyQualifiedClassName)
		end

		local result, err = pcall(requireWrapper, fullyQualifiedClassName)

		if not result then
			log.error('DefinitionTable:_requireClass(className) failed to' ..
				'require definition with className', className, err);
		elseif not self._validatorFn(def, className) then
			log.error('DefinitionTable:_requireClass(className) required a ' ..
				'definition that failed validation with className', className)
		else
			self._defs[className] = def
			return def
		end
end

local function transcodeXml(filename, fileContent)
	-- XmlViewTranscoder:transcodeView returns a pointer to a TranscodedView.
	local transcodedView = getXmlViewTranscoder():transcodeView(filename, fileContent)

	-- We then convert the underlying char pointer to a Lua string.
	local transcodedString = ffi.string(transcodedView.data, transcodedView.length)

	-- And subsequently delete the TranscodedView so that it doesn't leak.
	getXmlViewTranscoder():deleteTranscodedView(transcodedView)

	return transcodedString
end

--[[
-- Loads a given definition from the xml views search path.
-- It tries to load a file from a given filename and returns nil if it does
-- not succeed. As it is scanning the whole search path simply (and non-fatal)
-- indicate that the view has not been found in the current path.
--]]
function DefinitionTable:loadXmlView(path, className)
	-- convert the filename from 'view.myview' to 'view/myview.xml'
	local convName = string.gsub(className, '%.', '/')
	local filename = path .. convName .. '.xml'

	-- it is ok to not find a xml view, the search continues with the next path
	local fileContent = FileLoader.loadFile(filename)
	if (fileContent == nil) then
		return nil
	end

	-- transcode the XML to a Lua view definition
	local luaCode = transcodeXml(filename, fileContent)
	assert(luaCode ~= nil and luaCode ~= "", "Failed to transcode xml view: " .. className)

	-- execute the view definition and store it
	local result, err = loadstring(luaCode)

	if result == nil then
		self:_logErrorAndDumpLuaToFileIfEnabled(err, luaCode, className)
	else
		local viewDef = result()
		assert(viewDef ~= nil, "Failed to execute transcoded view: " .. className)

		if not self._validatorFn(viewDef, className) then
			log.error('DefinitionTable:loadXmlView(className) loaded a definition ' ..
				'that failed validation with className', className)
			return nil
		end

		self._defs[className] = viewDef
		return viewDef
	end
end

function DefinitionTable:_logErrorAndDumpLuaToFileIfEnabled(err, luaCode, className)
	local filename = className .. '_Generated.lua'

	local message = '\n\n' ..
		'Error running generated Lua code for ' .. className .. ':' ..
		'\n\n' ..
		'	' .. err ..
		'\n\n'

	if _dumpGeneratedLuaFilesOnError then
		message = message ..
			'Dumping generated Lua file to ' .. filename
	else
		message = message ..
			'Pass the `dump-generated-lua-views-on-error` flag ' ..
			'to enable dumping of the generated Lua file to disk.'
	end

	log.error(message .. '\n\n')

	if _dumpGeneratedLuaFilesOnError then
		local file = io.open(filename, 'w')
		io.output(file)
		io.write(luaCode)
		io.close(file)
	end
end

--[[
-- Loads a given definition from the xml views search path
--
--]]
function DefinitionTable:loadFromXmlSearchPath(className)
	local searchPath = self:getXmlSearchPath()
	local paths = plUtils.split(searchPath, ';', true)

	-- try to load the given xml file from the search path
	for i, path in ipairs(paths) do
		local handle = self:loadXmlView(path, className)
		if (handle ~= nil) then
			return handle
		end
	end

	-- no xml file found, try the native lua ones next
	return nil
end

--[[
-- Loads a given definition from the search path my className/key
--
--]]
function DefinitionTable:load(className)
	-- check if it already has been loaded
	if self:has(className) then
		return self._defs[className]
	end
	-- try to load the view from the xml definitions first
	local handle = self:loadFromXmlSearchPath(className)
	-- if it is not found/loaded, try Lua views next
	if (handle == nil) then
		handle = self:loadFromLuaSearchPath(className)
	end
	return handle
end

--[[
-- Unloads a given definition by className/key
--
--]]
function DefinitionTable:unload(className)
	if (self:has(className)) then
		self._defs[className] = nil
		package.loaded[className] = nil
		package.preload[className] = nil
		_G[className] = nil
	end

	return true;
end

--[[
-- Check if the table has a given definition by key/className
--
--]]
function DefinitionTable:has(className)
	return self._defs[className] ~= nil
end

--[[
-- Returns a definition by className. If it doesn't exist, nil is returned
--
--]]
function DefinitionTable:get(className)
	return self._defs[className]
end

--[[
-- Returns a count of the number of definitions in the table
--
--]]
function DefinitionTable:size()
	local size = 0
	for className, definition in pairs(self._defs) do
		size = size + 1
	end
	return size
end

return DefinitionTable
