-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ViewFactory = class()

--[[
	Creates a new view instance by name taking an option ViewHandlePtrUserData
	as an argument.
]]
function ViewFactory:createView(viewName, mediationGroup, options)
	local definition = loadView(viewName)
	return self:createViewWithDefinition(definition, mediationGroup, options)
end

function ViewFactory:createViewWithDefinition(definition, mediationGroup, options)
	local options = options or {}
	local properties = options.properties or {}
	local handle = options.handle

	local instance = definition.new()

	for propertyName, propertyValue in pairs(properties) do
		if instance:acceptsProperty(propertyName) then
			instance:setProperty(propertyName, propertyValue)
		end
	end

	-- Assert that all required view properties have been set by this point
	-- to ensure the view will behave in a predictable way
	instance:validateProperties()

	if not handle then
		handle = self:_createViewHandlePtrUserData(instance)
	end

	-- Handle Registration
	handle:assignLuaViewInstanceUuid(instance:getUuid())
	if mediationGroup ~= nil then
		handle:setMediationGroup(mediationGroup)
	end
	getViewComponentManager():assignJsObjectRetainer(handle)

	-- View Initialisation
	instance:setHandle(handle)
	instance:createResources()

	-- WARNING!!!!
	-- createMediator() is an FFI call that might trigger a call back into
	-- LuaJIT from native code (eg if an app makes ViewHandle calls in response
	-- to the JS MediatorFactoryCallbackBinding being fired).
	-- The call chain 
	--     JITted Lua -> native FFI -> more Lua
	-- is strictly forbidden and will probably cause LuaJIT to crash.
	-- Therefore we must ensure this call is not from JITted code.
	-- Another solution might be to make createMediator() a C-API binding
	-- rather than FFI. 

	jit.off(true)
	getViewComponentManager():createMediator(definition.typeName, handle)
	instance:onMediatorReady()

	return instance
end

function ViewFactory:_createViewHandlePtrUserData(instance)
	return ViewHandleSharedPtrUserData.new()
end

return ViewFactory
