-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

plugins.setPluginAlias('views', 'com.amazon.ignition.framework.views.logic')

local Context = plugins.views:lazyRequire('core.context.Context')

-- Fundamental Types
IElement = plugins.views:lazyRequire("core.types.IElement")
IView = plugins.views:lazyRequire("core.types.IView")
IDirective = plugins.views:lazyRequire("core.types.IDirective")

-- Global APIs
OpConversion = plugins.views:lazyRequire('core.utils.OpConversion')
OpsContainer = plugins.views:lazyRequire('utils.OpsContainer')

-- Global Utils
plugins.views:require('core.utils.ElementClassResolver')

-- Local Utils
local PreprocessFuncLib = plugins.views:lazyRequire('core.utils.PreprocessFunctionLibrary')
local DefinitionTable = plugins.views:lazyRequire('core.definitions.DefinitionTable')
local Timers = plugins.views:lazyRequire('utils.Timers')

--[[
-- Global 'Context' instance which provides the view factory, registry etc.
--
--]]
context = Context.new()

--[[
-- Global 'SignalBus' instance which provides a way of communicating inside
-- the lua state
--
 ]]
signalBus = plugins.lua:lazyRequire("signals.SignalBus").new()

--[[
-- Reference to the VCM controlling the state
--
--]]
local vcm

--[[
-- Reference to the XmlViewTranscoder instance
--
--]]
local transcoder

--[[
-- Timers used for tracking time taken to load views, flush ops etc.
--
 ]]
local timers = Timers.new()

--[[
-- Internal method for simple iteration of the dependency array
--
--]]
local function checkOrLoad(check)
	for index, className in pairs(check.missing) do
		if not check.defTable:has(className) then
			local def = check.load(className)
			if (def == nil) then
				table.insert(check.result, className)
			end
		end
	end
end

--[[
-- Attempts to load all the dependencies in the given table.
-- If any fail to load, their names are returned in the results table
--
--]]
function autoloadDependencies (deps)
	local result = {views = {}, directives = {}}

	local viewCheck =
	{
		missing = deps.views,
		defTable = context:getViewDefinitionTable(),
		result = result.views,
		load = function (className)
			return loadView(className)
		end
	}

	local directiveCheck =
	{
		missing = deps.directives,
		defTable = context:getDirectiveDefinitionTable(),
		result = result.directives,
		load = function (className)
			return context:getDirectiveDefinitionTable():load(className)
		end
	}

	if deps.views ~= nil then
		checkOrLoad(viewCheck)
	end

	if deps.directives ~= nil then
		checkOrLoad(directiveCheck)
	end

	return result
end

--[[
-- Checks to see if the result returned via `autoloadDependencies` is missing
-- any dependencies.
--
-- Returns false if nothing is missing
--]]
local function hasMissingDependencies(missing, className)
	local missingStr, hasMissing = false

	local function checkMissing(source, className)
		if (table.getn(source) > 0) then
			missingStr = (missingStr or "") .. className .. "[" .. table.concat(source, ", ") .. "] "
			hasMissing = true
		end
	end

	checkMissing(missing.views, "Views")
	checkMissing(missing.directives, "Directives")

	if (hasMissing) then
		log.error("Definition for View named", className, "is missing the following deps", missingStr)
	end

	return hasMissing
end

--[[
-- Loads missing view dependencis
--]]
local function loadViewDependencies(viewDef, definitionTable)

	-- View def un-loadable exit
	if (viewDef == nil) then
		return nil
	end

	-- Check if the definition is missing dependencies
	local missing = autoloadDependencies(viewDef:getDependencies())
	if (hasMissingDependencies(missing, viewDef.typeName)) then
		definitionTable:unload(viewDef.typeName)
		return nil
	end

	-- View has loaded and all deps have been satisfied
	return viewDef
end

--[[
-- Loads a single view definition by className
--]]
function loadView(className)
	local timerIndex = timers.LOAD_VIEW:startTiming()

	local definitionTable = context:getViewDefinitionTable()
	local viewDef = definitionTable:load(className)
	local result = loadViewDependencies(viewDef, definitionTable)

	timers.LOAD_VIEW:endTiming(timerIndex)

	return result
end

--[[
-- Unloads a single view definition by className
--
-- C++ Method signature:
-- BindingResultType unloadView(const std::string);
--]]
function unloadView(className)
	return context:getViewDefinitionTable():unload(className)
end

--[[
-- Sets the Lua view definition loading search path
--
-- C++ Method signature
-- BindingResultType setLuaSearchPath(const std::string);
--]]
function setLuaSearchPath(path)
	context:getViewDefinitionTable():setLuaSearchPath(path)
	context:getDirectiveDefinitionTable():setLuaSearchPath(path)
end

--[[
-- Gets the Lua view definition loading search path
--
-- C++ Method signature
-- BindingResultType getLuaSearchPaths(std::string*);
--]]
function getLuaSearchPath()
	-- Assumes that the search paths in viewDefs and directiveDefs match
	return context:getViewDefinitionTable():getLuaSearchPath()
end

--[[
-- Sets the Xml view definition loading search path
--
-- C++ Method signature
-- BindingResultType setXmlSearchPath(const std::string);
--]]
function setXmlSearchPath(path)
	context:getViewDefinitionTable():setXmlSearchPath(path)
	context:getDirectiveDefinitionTable():setXmlSearchPath(path)
end

--[[
-- Gets the Xml view definition loading search path
--
-- C++ Method signature
-- BindingResultType getXmlSearchPath(std::string*);
--]]
function getXmlSearchPath()
	-- Assumes that the search paths in viewDefs and directiveDefs match
	return context:getViewDefinitionTable():getXmlSearchPath()
end

--[[
-- Returns a directive definition by className if it exists.
-- If it doesn't exist nil is returned.
--
--]]
function getLoadedDirective(className)
	assert(className ~= nil, 'getLoadedDirective: className must not be nil')
	local def = context:getDirectiveDefinitionTable():get(className)
	if def == nil then
		error('Directive with className ' .. className .. ' has not been loaded')
	end
	return def
end

--[[
-- Returns a view definition by className if it exists.
-- If it doesn't exist nil is returned.
--
--]]
function getLoadedView(className)
	local def = context:getViewDefinitionTable():get(className)
	if def == nil then
		error('View with className ' .. className .. ' has not been loaded')
	end
	return def
end

function getNumLoadedViews()
	return context:getViewDefinitionTable():size()
end

function getNumInstantiatedViews()
	return context:getViewRegistry():getCount()
end

--[[
-- Sets the ViewComponentManager instance
--
--]]
function setViewComponentManager(newVcm)
	vcm = newVcm;
	return true
end

--[[
-- Gets the ViewComponentManager instance
--
--]]
function getViewComponentManager()
	if vcm == nil then
		error('getViewComponentManager failed. No VCM Set')
	end
	return vcm
end

--[[
-- Sets the XmlViewTranscoder instance
--
--]]
function setXmlViewTranscoder(value)
	transcoder = value
	return true
end

--[[
-- Gets the XmlViewTranscoder instance
--
--]]
function getXmlViewTranscoder()
	if transcoder == nil then
		error('getXmlViewTranscoder failed. No transcoder set.')
	end
	return transcoder
end

--[[
-- Calls flush ops on all views currently in existence
--]]
function flushOps()
	local timerIndex = timers.FLUSH_OPS:startTiming()

	context:getTickableRegistry():tick()

	context:getSceneNodeOpsQueue():flushOps()

	timers.FLUSH_OPS:endTiming(timerIndex)
end

--[[
-- Retrieves a pre-process function from a library of existing functions
--
--]]
function getPreprocessFunction(functionName)
    local func = PreprocessFuncLib.getFunction(functionName)
    if func == nil then
        error('Failed to retrieve function: ' .. functionName)
    end
    return func
end

--[[
-- Adds a function to the pre-process function library, allowing it to be
-- retrieved via getPreprocessFunction
--
--]]
function addPreprocessFunction(functionName, func)
    if not PreprocessFuncLib.addFunction(functionName, func) then
        error('Cannot add duplicate function! ' .. functionName .. ' already exists')
    end
end


function getViewFromRegistry(uuid)
	return context:getViewRegistry():getViewInstanceByUuid(uuid)
end

function setI18nDictionary(keysToValues)
	return context:getI18nDictionary():setValues(keysToValues)
end

function enableDumpingOfGeneratedLuaFilesOnError()
	DefinitionTable.setDumpGeneratedLuaFilesOnError(true)
end

function getTimings()
	local timings = timers:getAllTimings()
	timers:resetAllTimers()
	return timings
end

-- TODO (pcowburn) This is a ridiculous hack :D basically, currently preprocessor
-- functions only apply to routed attributes - not on static ones. In order to
-- support the i18n() function in both routed and static attributes, right now
-- we have make a global function with the same name that can return the value
-- from the dictionary.
--
-- The fact that you can actually call a function from a static attribute is
-- probably not great security wise either - I've added a Jira for that here:
-- https://jira-int.amazondcl.com/browse/IG-309.
--
-- For now, this makes the feature work, which does mean that Doppelganger can
-- start using it at least. But we probably need to spend a bit of time adjusting
-- things so that preprocessor functions can be used in static attributes. I've
-- added a Jira for that too https://jira-int.amazondcl.com/browse/IG-310.
function i18n(key)
	return context:getI18nDictionary():getValue(key)
end

function disableSceneGraphChangesFromLua()
	SceneNodeWrapper.ENABLE_SCENE_GRAPH_CHANGES_FROM_LUA = false
end

function enableSceneGraphChangesFromLua()
	SceneNodeWrapper.ENABLE_SCENE_GRAPH_CHANGES_FROM_LUA = true
end

-- Facilitates the creation of a new ViewInstance.
-- This function is intended to be called from the C++ bindings
-- Internal view creation should be done via the ViewFactory on the Context
function createViewInstance(viewName, mediationGroup, handle)
	return context:getViewFactory():createView(viewName, mediationGroup, { handle = handle })
end
