-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local AbstractViewportPositioner = plugins.views:lazyRequire('views.infiniteList.view.positioning.AbstractViewportPositioner')
local IteratorOptions = plugins.views:lazyRequire("properties.iterator.IteratorOptions")

local FixedHighlightViewportPositioner = class(AbstractViewportPositioner,
	function(self, highlightedPosition)
		if (highlightedPosition == nil) then
			error('The highlightedPosition argument is required, and must be a number')
		end

		self._highlightedPosition = highlightedPosition
	end)

function FixedHighlightViewportPositioner:_getStartAndEndIndices(
		highlightedIndex, viewport)
	local startIndex = highlightedIndex - self._highlightedPosition
			- viewport:getPreBufferSize() + 1
	local endIndex = startIndex + viewport:getSize() - 1

	return startIndex, endIndex
end

function FixedHighlightViewportPositioner:setHighlightedIndex(viewport, index)
	AbstractViewportPositioner.setHighlightedIndex(self, viewport, index)

	local diffFromStart = index - viewport:getStartIndex()
	if diffFromStart >= self._highlightedPosition then
		local preBuffer = diffFromStart + 1 - self._highlightedPosition
		viewport:setNumItemsInPreBuffer(preBuffer)
	else
		viewport:setNumItemsInPreBuffer(0)
	end
end

function FixedHighlightViewportPositioner:shouldReposition(viewport)
	local highlightedIndex = viewport:getHighlightedIndex()
	local previousHighlightedIndex = viewport:getPreviousHighlightedIndex()

	local activeIndex =
			highlightedIndex - viewport:getStartIndex() + 1
			- viewport:getNumItemsInPreBuffer()
	local previousActiveIndex =
			previousHighlightedIndex - viewport:getPreviousStartIndex() + 1
			- viewport:getPreviousNumItemsInPreBuffer()

	return ((highlightedIndex ~= previousHighlightedIndex)
			and (activeIndex == previousActiveIndex))
end

function FixedHighlightViewportPositioner:reposition(
		viewport, viewportViews, iterator, options)
	local oldHighlightedIndex = viewport:getPreviousHighlightedIndex()
	local itemInFirstPosition = viewport:getNumItemsInPreBuffer() + 1

	if oldHighlightedIndex ~= nil then
		local highlightedIndex = viewport:getHighlightedIndex()
		options:addOption("positionDelta", oldHighlightedIndex - highlightedIndex)
		options:addOption("wrappingEnabled", true)
	end

	options:addOption("itemInFirstPosition", itemInFirstPosition)

	iterator:iterate(viewportViews, options)
end

return FixedHighlightViewportPositioner
