-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local Signal = plugins.views:lazyRequire('signals.Signal')

-- Represents the visible part of the data set, in terms of its start and end
-- indices within the whole set. The start and end indices themselves are in
-- turn driven by the highlighted index - as it moves, the start and end indices
-- are updated if necessary in order to keep it in view.
local Viewport = class(function(self, size, preBufferSize)
	if type(size) ~= 'number' then
		error('The size argument is required, and must be a number')
	end

	self._size = size
	self._startIndex = nil
	self._endIndex = nil
	self._highlightedIndex = nil

	self._previousStartIndex = nil
	self._previousEndIndex = nil
	self._previousHighlightedIndex = nil

	self._preBufferSize = preBufferSize or 0
	self._numItemsInPreBuffer = 0
	self._previousNumItemsInPreBuffer = 0

	self.positionChanged = Signal.new()

	self:_updateDirectionOfTravel()
end)

Viewport.DIRECTION_FORWARDS = 'forwards'
Viewport.DIRECTION_BACKWARDS = 'backwards'

function Viewport:getSize()
	return self._size
end

function Viewport:getHighlightedIndex()
	return self._highlightedIndex
end

function Viewport:getPreviousHighlightedIndex()
	return self._previousHighlightedIndex
end

function Viewport:setHighlightedIndex(index)
	self:_updateDirectionOfTravel(index)

	self._previousHighlightedIndex = self._highlightedIndex
	self._highlightedIndex = index

	if self._highlightedIndex ~= self._previousHighlightedIndex then
		self.positionChanged:dispatch()
	end
end

function Viewport:getPreBufferSize()
	return self._preBufferSize
end

function Viewport:setNumItemsInPreBuffer(value)
	if value > self._preBufferSize then
		error("Viewport preBuffer can have at most " .. self._preBufferSize
				.. " elements. Trying to set it to " .. value)
	end

	self._previousNumItemsInPreBuffer = self._numItemsInPreBuffer
	self._numItemsInPreBuffer = value
end

function Viewport:getNumItemsInPreBuffer()
	return self._numItemsInPreBuffer
end

function Viewport:getPreviousNumItemsInPreBuffer()
	return self._previousNumItemsInPreBuffer
end

function Viewport:_updateDirectionOfTravel(index)
	if self._highlightedIndex == nil then
		self._directionOfTravel = Viewport.DIRECTION_FORWARDS
	elseif index > self._highlightedIndex then
		self._directionOfTravel = Viewport.DIRECTION_FORWARDS
	elseif index < self._highlightedIndex then
		self._directionOfTravel = Viewport.DIRECTION_BACKWARDS
	end
end

function Viewport:getStartIndex()
	return self._startIndex
end

function Viewport:getPreviousStartIndex()
	return self._previousStartIndex
end

function Viewport:setStartIndex(index)
	self._previousStartIndex = self._startIndex
	self._startIndex = index
end

function Viewport:getEndIndex()
	return self._endIndex
end

function Viewport:getPreviousEndIndex()
	return self._previousEndIndex
end

function Viewport:setEndIndex(index)
	self._previousEndIndex = self._endIndex
	self._endIndex = index
end

function Viewport:getDirectionOfTravel()
	return self._directionOfTravel
end

function Viewport:dispose()
	self.positionChanged:dispose()
end

return Viewport