-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ViewPropertiesRegistry = plugins.views:lazyRequire("core.definitions.ViewPropertiesRegistry")
local IView = plugins.views:lazyRequire('core.types.IView')
local ModelDiffer = plugins.views:lazyRequire('utils.ModelDiffer')
local FocusPolicy = plugins.views:lazyRequire('properties.values.FocusPolicy')

local cjson = lazyRequire('cjson')

local BINDING_UPDATED_SIGNAL_NAME = 'bindingUpdated'

local ElementView = class(ViewClass(IView, function(self, handle)
	IView._ctor(self, handle)

	-- We need to keep a reference to the current model so that the lua
	-- object is not garbage collected, otherwise it would be removed from
	-- the JSObjectRetainer (if coming from javascript)
	self._currentModel = nil
	self._lastFocusedDescendantNodeId = nil

	self._instanceProperties = {}

	self:addSignal(BINDING_UPDATED_SIGNAL_NAME)
end))

local _recursivelyCreateOpsForStaticProps

ElementView.typeName = 'ElementView'

ElementView.elements = {}

ElementView.propertyDefinitions = ViewPropertiesRegistry.mergePropertyDefinitions(
	IView.propertyDefinitions,
	{
		forwardFocusToChildren =
		{
			default = false
		},
		focusPolicy =
		{
			default = FocusPolicy.DEFAULT
		}
	}
)

ElementView._mustHaveRootElement = false

function ElementView:createResources()
	self:_checkWhetherRootElementIsRequiredButNotPresent()
	self:_preprocessElements()
	self:_createElements()
	self:_applyStaticProps()
	self:_setViewNameOnRootElement()

	local signal = self:addBubblingSignal("internalFocusLost")
	signal:add(self._onFocusLost, self)

	if self.defaultFocusedElement ~= nil then
		self:getHandle():setDefaultFocusedNode(
				self.defaultFocusedElement)
	end

	return true
end

function ElementView:getCurrentModel()
	return self._currentModel
end

function ElementView:_checkWhetherRootElementIsRequiredButNotPresent()
	if  (self._mustHaveRootElement == true)
	and (self.elements[1] == nil)
	then
		self:_error(self.typeName .. ' must have a root element')
	end
end

function ElementView:_preprocessElements()
	for i = 1, #self.elements do
		self.elements[i].index = i
	end
end

function ElementView:_createElements()
	self:_eachElement(function(index, element)
		if (element.type == IView.TYPE_VIEW) then
			self:_createViewElement(element.index, element.className,
				{
					name = element.name,
					properties = self:_getStaticViewPropsForViewAtIndex(element.index)
				})
		elseif (element.type == IView.TYPE_DIRECTIVE)
		    or (element.type == IView.TYPE_VIEW_FRAGMENT) then
			self:_createInlinedElement(element, element.name)
		else
			self:_error(
				'Unknown element type in _createElements: ' .. tostring(element.type),
				element,
				element.type)
		end

		if element.parent ~= nil then
			self:_attachElements(element.parent, element.index)
		end

		if element.mask ~= nil then
			self:_applyElementMask(element.mask, index)
		end
	end)
end

function ElementView:_createViewElement(index, className, options)
	local view = context:getViewFactory():createView(
		className,
		options.mediationGroup or self:getMediationGroup(),
		{
			handle = options.handle,
			properties = options.properties
		})

	self:setChildElement(index, view)

	if options.name ~= nil then
		view:setName(options.name)
	end

	context:getNodesRegistry():add(
			view:getRootNode():getSceneNode():getId(),
			view)

	return view
end

function ElementView:_createInlinedElement(element, name)
	local instance = self:_createViewFromElementDefinition(element)

	local nodeId = instance:getRootNode():getSceneNode():getId()
	if element.index == 1 then
		context:getNodesRegistry():add(nodeId, self)
	else
		context:getNodesRegistry():add(nodeId, instance)
	end

	self:setChildElement(element.index, instance)

	if name ~= nil then
		local node = instance:getRootNode():getSceneNode()
		node:setName(element.name)
	end
end

function ElementView:_createViewFromElementDefinition(element)
	local definition = self:_getDefinitionByElement(element)
	if(element.type == IView.TYPE_DIRECTIVE) then
		local directive = definition.new()
		directive:createResources()
		return directive
	else
		return context:getViewFactory():createViewWithDefinition(
			definition,
			self:getMediationGroup())
	end
end

function ElementView:_getDefinitionByElement(element)
	local definition

	if element.type == IView.TYPE_DIRECTIVE then
		definition = getLoadedDirective(element.className)

	elseif element.type == IView.TYPE_VIEW_FRAGMENT then
		definition = self:getFragmentDefinition(element.className)

	elseif element.type == IView.TYPE_VIEW then
		definition = getLoadedView(element.className)

	else
		self:_error(
			'Unknown element type in _getDefinitionByElement: ' .. element.type,
			element,
			element.type)
	end

	if definition == nil then
		self:_error(
			'Unknown element className in _getDefinitionByElement: ' .. element.className,
			element,
			element.className)
	end

	return definition
end

function ElementView:_attachElements(parentIndex, childIndex)
	local parentNode = self:_getRootNodeByIndex(parentIndex)
	local childNode = self:_getRootNodeByIndex(childIndex)

	assert(parentNode ~= nil, '_attachElements: parentNode must not be nil')
	assert(childNode ~= nil, '_attachElements: childNode must not be nil')

	context:getSceneNodeOpsQueue():queueOp(
		parentNode:getSceneNode(),
		"addChild",
		childNode:getSceneNode())
end

function ElementView:_applyElementMask(maskIndex, subjectIndex)
	local maskNode = self:_getRootNodeByIndex(maskIndex)
	local subjectNode = self:_getRootNodeByIndex(subjectIndex)

	assert(maskNode ~= nil, '_applyElementMask: maskNode must not be nil')
	assert(subjectNode ~= nil, '_applyElementMask: subjectNode must not be nil')

	context:getSceneNodeOpsQueue():queueOp(
		subjectNode:getSceneNode(),
		"addMaskNode",
		maskNode:getSceneNode())
end

function ElementView:_eachElement(callback)
	for index = 1, #self.elements do
		callback(index, self.elements[index])
	end
end

function ElementView:_getStaticViewPropsForViewAtIndex(index)
	local element = self.elements[index]
	local staticViewProps = {}

	if element.staticProps ~= nil then
		for propertyName, propertyValue in pairs(element.staticProps) do
			staticViewProps[propertyName] = propertyValue
		end
	end

	return staticViewProps
end

function ElementView:_applyStaticProps()
	self:_eachElement(function (index, element)
		local props = element.staticProps

		if props ~= nil then
			local childView = self:getChildElement(index)

			if (element.type == IView.TYPE_VIEW)
			or (element.type == IView.TYPE_VIEW_FRAGMENT) then
				-- Create a temporary ops container based on the static properties
				-- if supplied.
				local ops = self:_createOpsForStaticProps(props)

				-- If a temporary ops container has been created we need to pass
				-- the ops to the child view's preflight and update methods so
				-- that they're immediately applied.
				if ops ~= nil then
					childView:preflight(ops)
					childView:update(ops)
				end

			elseif element.type == IView.TYPE_DIRECTIVE then
				childView:setProperties(self:_staticPropsToPropArray(props))

			end
		end
	end)
end

function ElementView:update(ops)
	self:_saveInstanceProperties(ops)

	return true
end

function ElementView:_saveInstanceProperties(ops)
	for propertyName, propertyValue in pairs(ops:getLocals()) do

		if propertyValue.type == OpsContainer.OPERATION_CREATE or
			propertyValue.type == OpsContainer.OPERATION_UPDATE then
			self._instanceProperties[propertyName] = propertyValue.value
		elseif propertyValue.type == OpsContainer.OPERATION_DELETE then
			self._instanceProperties[propertyName] = nil
		end
	end
end

function ElementView:getPropertyAsJson(propertyName)
	local prop

	if self:acceptsProperty(propertyName) then
		prop = self:getProperty(propertyName)
	end

	prop = prop or self._instanceProperties[propertyName]

	return cjson.encode(prop)
end

function ElementView:_setViewProperties(view, localOps)
	if localOps then
		for property, propertyData in pairs(localOps) do
			if view:acceptsProperty(property) then
				view:setProperty(property, propertyData.value)
			end
		end
	end
end

function ElementView:_createOpsForStaticProps(staticProps)
	return _recursivelyCreateOpsForStaticProps(OpsContainer.new(), staticProps)
end

-- Recursively walks down the supplied list of static properties, creating ops
-- containers for each level of nesting and local ops for each property value.
function _recursivelyCreateOpsForStaticProps(ops, staticProps)
	for key, value in pairs(staticProps) do
		if type(value) == 'table' then
			-- Create child ops container
			local childOps = ops:addChild(key)
			childOps:setType(OpsContainer.OPERATION_CREATE)

			_recursivelyCreateOpsForStaticProps(childOps, value)
		else
			-- Create local value
			ops:setLocal(key, value)
		end
	end

	return ops
end

function ElementView:_staticPropsToPropArray(staticProps)
	local propArray = {}

	for name, value in pairs(staticProps) do
		table.insert(propArray,
		{
			name = name,
			value = value
		})
	end

	return propArray
end

function ElementView:_setViewNameOnRootElement()
	self:getRootNode():getSceneNode():setViewName(self.typeName)
end

function ElementView:updateModel(model)
	self:pushOps(self:_calculateModelDiff(model))

	return true
end

function ElementView:pushOps(ops)
	self:preflight(ops)
	self:update(ops)
end

function ElementView:preflight(ops)
	if ops:getModel() ~= nil then
		self._currentModel = ops:getModel()
	end

	self:_dispatchBindingUpdatedSignalIfBeingListenedTo(ops)

	return true
end

-- When a binding update signal is dispatched, usually the name of the binding
-- is taken from the XML property the value was bound to. For example, in the
-- following XML snippet the binding name would be `rating`:
--
--     <view class="RatingsView" rating="@{imdb.score}" />
--
-- However, there are some cases where values are bound to views without being
-- delivered via a user-specified binding name. Such is the case with children
-- of an IteratingView, where the corresponding item from the array being iterated
-- is pushed directly to the iteratee. In these cases the DEFAULT_BINDING_NAME
-- constant is used as the name of the binding when the bindingUpdated signal
-- is dispatched.
local DEFAULT_BINDING_NAME = 'model'

local BindingUpdatedPayloadType =
{
	JS_OBJECT_ID = 'BindingUpdatedPayloadType.JS_OBJECT_ID',
	PRIMITIVE_TYPE = 'BindingUpdatedPayloadType.PRIMITIVE_TYPE'
}

function ElementView:_dispatchBindingUpdatedSignalIfBeingListenedTo(ops)
	local signal = self:getSignal(BINDING_UPDATED_SIGNAL_NAME)

	if signal and signal:getNumListeners() > 0 then

		-- Handles cases where the object is being pushed directly to the view,
		-- rather than to an individual binding name. Described in more detail
		-- alongside the comment for DEFAULT_BINDING_NAME.
		if ops:getObjectId() ~= nil then
			self:_dispatchBindingUpdatedSignalWithParams(DEFAULT_BINDING_NAME, ops)
		else
			-- Handles cases where objects are being pushed to an indiviudal
			-- bound property.
			for name, childOp in pairs(ops:getChildren()) do
				self:_dispatchBindingUpdatedSignalWithParams(name, childOp)
			end
		end

		-- Handles cases where the value is a primitive type such as a string,
		-- or a number, rather than a JS object.
		for name, localOp in pairs(ops:getLocals()) do
			self:_dispatchBindingUpdatedSignalWithParams(name, localOp.value)
		end
	end
end

function ElementView:_dispatchBindingUpdatedSignalWithParams(name, value)
	local signalParams = ReflectableValueVectorUserData.new()

	local nameParam = ReflectableValueUserData.new()
	nameParam:setString(name)

	local jsObjectIdParam = ReflectableValueUserData.new()

	if self:_isJsObject(value) then
		jsObjectIdParam:setId(value:getObjectId())
	else
		jsObjectIdParam:set(value)
	end

	signalParams:push(nameParam)
	signalParams:push(jsObjectIdParam)

	self:getSignal(BINDING_UPDATED_SIGNAL_NAME):dispatch(signalParams)
end

function ElementView:_isJsObject(value)
	return type(value) == 'table' and value:getObjectId() ~= nil
end

function ElementView:getCurrentOps()
	return self._opsQueue[1]
end

function ElementView:_calculateModelDiff(model)
	local success, result = pcall(function()
		if self._modelDiffer == nil then
			self._modelDiffer = ModelDiffer.new()

			-- There can be cases where the view already has a model, but hasn't
			-- actually performed a diff by itself yet - this is the case when
			-- a model is first propagated down to the view via pushOps() from an
			-- ancestor view, but then later someone calls updateModel() on the
			-- view directly.
			--
			-- In this case we have to prime the model differ by supplying it the
			-- existing model. This is done so that when diff() is called again later
			-- with a new model value, the differ has something to compare it against.
			if self._currentModel ~= nil then
				self._modelDiffer:diff(self._currentModel)
			end
		end

		local deletedObjectIds = {}
		local diff = self._modelDiffer:diff(model, deletedObjectIds)

		return diff
	end)

	if success then
		return result
	else
		self:_error('Error performing model diff: ' .. result)
	end
end

function ElementView:_createJsObjectPlaceholderFromOps(opsContainer)
	return self:_createJsObjectPlaceholder(opsContainer:getObjectId())
end

function ElementView:_createJsObjectPlaceholder(objectId)
	if objectId == nil then
		self:_error('Cannot build a JS object placeholder without an object id')
	end

	local signalParams = ReflectableValueVectorUserData.new()
	local jsObjectId = ReflectableValueUserData.new()

	jsObjectId:setId(objectId)
	signalParams:push(jsObjectId)

	return signalParams
end

function ElementView:dispose()
	self:_disposeElements()

	return IView.dispose(self)
end

function ElementView:_disposeElements()
	for index = 1, #self._childElements do
		self:getChildElement(index):dispose()
		self:setChildElement(index, nil)
	end
end

function ElementView:getRootNode()
	local element = self:getChildElement(1)
	return element and element:getRootNode()
end

function ElementView:_getLastFocusedDescendantNode()
	if self._lastFocusedDescendantNodeId ~= nil then

		local nodePtr = SceneNodePtr.fromSceneNodeUuid(
			self._lastFocusedDescendantNodeId)

		if nodePtr ~= nil then
			return nodePtr:getSceneNode()
		end
	end

	return nil
end

-- This method should only be overridden when a view wants to forward its focus
-- to a designated child without exception
function ElementView:getFocusForwardedNode()
	if self:_isFocusMemoryEnabled() and	self:_canFocusLastFocusedDescendantNode() then
		return self:_getLastFocusedDescendantNode()
	elseif self:getProperty("forwardFocusToChildren") then
		return self:getFocusCandidate()
	end
end

function ElementView:_isFocusMemoryEnabled()
	return self:getProperty("focusPolicy") == FocusPolicy.REMEMBER_FOCUS
end

local function getFocusCandidateRec(element)
	for i = 1, element:getNumChildElements() do
		local child = element:getChildElement(i)
		local focusCandidate

		if child:instanceOf(ElementView) then
			focusCandidate = child:getFocusCandidate()
		else
			focusCandidate = getFocusCandidateRec(child)
		end

		if not focusCandidate then
			local childRootNode = child:getRootNode()
			if childRootNode and childRootNode:getSceneNode():getFocusable() == true then
				focusCandidate = childRootNode:getSceneNode()
			end
		end

		if focusCandidate then
			return focusCandidate
		end
	end
end

-- This method iterates on all node hierarchy in order to find the first focus candidate
-- It should be overridden by views wishing to implement custom focus forwarding
function ElementView:getFocusCandidate()
	return getFocusCandidateRec(self)
end

function ElementView:interceptFocusGain(oldNode, newNode, direction, callingView)
	local chosenNode = nil

	if self:_isFocusMemoryEnabled() and
			self:_canFocusLastFocusedDescendantNode() and
			not self:_isDescendant(oldNode) then
		chosenNode = self:_getLastFocusedDescendantNode()
	else
		chosenNode = IView.interceptFocusGain(
			self, oldNode, newNode, direction, self)
	end

	if chosenNode ~= nil then
		return chosenNode
	else
		return newNode
	end
end

function ElementView:_canFocusLastFocusedDescendantNode()
	local node = self:_getLastFocusedDescendantNode()
	return node ~= nil and self:_isDescendant(node, true)
end

function ElementView:_isDescendant(node, includeRoot)
	includeRoot = includeRoot or false
	local isDescendant = includeRoot and node == self:getRootNode():getSceneNode()

	if node then
		local nodeParent = node
		while nodeParent:hasParent() do
			nodeParent = nodeParent:getParent()

			local selfNode = self:getRootNode():getSceneNode()
			if selfNode:getId() == nodeParent:getId() then
				isDescendant = true
				break
			end
		end
	end
	return isDescendant
end

function ElementView:_onFocusLost(node)
	if node ~= nil then
		self._lastFocusedDescendantNodeId = node:getId()
	else
		self._lastFocusedDescendantNodeId = nil
	end
end

function ElementView:hasFocus()
	local rootNode = self:getRootNode()
	if not rootNode then
		return false
	end

	rootNode = rootNode:getSceneNode()
	return rootNode:hasTag("focused") or rootNode:hasTag("descendant-focused")
end

function ElementView:_getRootNodeByIndex(index)
	local element = self:getChildElement(index)

	if element == nil then
		self:_error('No child element exists at index ' .. index)
	else
		return element:getRootNode()
	end
end

return ElementView
