-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ViewPropertiesRegistry = plugins.views:lazyRequire("core.definitions.ViewPropertiesRegistry")

local ElementView = ViewClass("ElementView")
local RoutingView = ViewClass("RoutingView")
local DynamicLoaderView = class(RoutingView)

DynamicLoaderView.typeName = 'DynamicLoaderView'

-- The name of the ops keyword which specifies the name of the view which
-- is to be loaded.
DynamicLoaderView.viewNameOpsKeyword = 'viewName'
-- TODO (pcowburn) Provided for backwards compatibility until all apps have
-- switched over to use the non-underscore prefixed version.
DynamicLoaderView.legacyViewNameOpsKeyword = '__viewName'

-- The name of the ops keyword which specifies the name of the mediation group of
-- the view to be loaded
DynamicLoaderView.mediationGroupOpsKeyword = 'mediationGroup'

-- Index that the loaded view will be placed at in the _childElements array.
DynamicLoaderView.loadedViewIndex = 2

-- DynamicLoaderViews have to have a container node as their root element so
-- that the loaded view has something to be attached to.
DynamicLoaderView._mustHaveRootElement = true

DynamicLoaderView.propertyDefinitions = ViewPropertiesRegistry.mergePropertyDefinitions(
	RoutingView.propertyDefinitions,
	{
		viewName =
		{
			default = "",
			onChange = function(self, oldValue, newValue)
				self:_createLoadedViewIfChanged(newValue, self:_getMediationGroup())
			end
		},
		mediationGroup =
		{
			default = nil,
			onChange = function(self, oldValue, newValue)
				self:_createLoadedViewIfChanged(self:_getViewName(), newValue)
			end
		}
	}
)

DynamicLoaderView.elements =
{
	{
		className = 'ContainerNodeDirective',
		type = ElementView.TYPE_DIRECTIVE,
		parent = nil
	}
}

function DynamicLoaderView:createResources()
	self._pendingOps = {}

	return RoutingView.createResources(self)
end

function DynamicLoaderView:getFocusForwardedNode()
	return self:getLoadedView():getRootNode():getSceneNode()
end

function DynamicLoaderView:preflight(ops)
	self:_createLoadedViewIfChanged(self:_getViewName(ops), self:_getMediationGroup(ops))

	if self._loadedView ~= nil then
		return self._loadedView:preflight(ops)
	else
		return true
	end
end

function DynamicLoaderView:_getViewName(ops)
	local viewNameProp = (ops and self:_getNonDeletedLocal(ops, self.viewNameOpsKeyword))
			or (ops and self:_getNonDeletedLocal(ops, self.legacyViewNameOpsKeyword))
			or self:getProperty("viewName")

	if viewNameProp ~= nil then
		return viewNameProp.value
	else
		return nil
	end
end

function DynamicLoaderView:_getMediationGroup(ops)
	local mediationGroupProp = (ops and self:_getNonDeletedLocal(ops, self.mediationGroupOpsKeyword))
			or self:getProperty("mediationGroup")

	if mediationGroupProp ~= nil then
		return mediationGroupProp.value
	else
		return nil
	end
end

function DynamicLoaderView:_getNonDeletedLocal(ops, name)
	local localOp = ops:getLocal(name)
	if localOp ~= nil and localOp.type ~= OpsContainer.OPERATION_DELETE then
		return localOp
	else
		return nil
	end
end

function DynamicLoaderView:_createLoadedViewIfChanged(newViewName, newMediationGroup)
	if ((newViewName ~= nil) and (newViewName ~= self._viewNameToLoad)) or
			(newMediationGroup ~= self._mediationGroupToLoadFrom) then
		self:_createLoadedView(newViewName, newMediationGroup)
	end
end

function DynamicLoaderView:_createLoadedView(viewName, mediationGroup)
	local autoloadResults = autoloadDependencies(
		{
			views = { viewName }
		})

	if table.getn(autoloadResults.views) > 0 then
		self:_error('View does not exist: ' .. autoloadResults.views[1])
	end

	if self._loadedView ~= nil then
		self:_detachLoadedView(self._loadedView)

		context:getSceneNodeOpsQueue():queueCallback(
			self._loadedView.dispose, self._loadedView)
	end

	self._viewNameToLoad = viewName
	self._mediationGroupToLoadFrom = mediationGroup
	self._loadedView = self:_createViewElement(self.loadedViewIndex, viewName,
		{
		 mediationGroup = mediationGroup
		})

	self:_processPendingOps()

	self:_attachLoadedView(self._loadedView)
end

function DynamicLoaderView:update(ops)
	if self._loadedView ~= nil then
		self:_performUpdateOnLoadedViewAndSelf(ops)
	else
		table.insert(self._pendingOps, ops)
	end
end

function DynamicLoaderView:_processPendingOps()
	if #self._pendingOps > 0 then
		for i, ops in ipairs(self._pendingOps) do
			self._loadedView:preflight(ops)
			self:_performUpdateOnLoadedViewAndSelf(ops)
		end
		self._pendingOps = {}
	end
end

function DynamicLoaderView:_performUpdateOnLoadedViewAndSelf(ops)
	self._loadedView:update(ops)

	-- DynamicLoadedView cannot itself contain aggregates and so applying
	-- aggregates that are intended for the loaded view actually leads to
	-- unexpected behaviour - as an example, the loaded view typically has
	-- many more elements than the DynamicLoaderView and so attempting to
	-- apply aggregates leads to an out of bounds exception when iterating
	-- the elements table. To get a clean set of ops to apply we clear the
	-- aggregates table before calling update() here.
	ops:clearAggregates()
	RoutingView.update(self, ops)
end

function DynamicLoaderView:_attachLoadedView(view)
	self:_addOrRemoveLoadedViewFromGraph('addChild', view)
end

function DynamicLoaderView:_detachLoadedView(view)
	self:_addOrRemoveLoadedViewFromGraph('removeChild', view)
end

function DynamicLoaderView:_addOrRemoveLoadedViewFromGraph(addOrRemoveChild, view)
	local selfNode = self:getRootNode()
	local childNode = view:getRootNode()

	assert(selfNode ~= nil, 'selfNode must not be nil')
	assert(childNode ~= nil, 'childNode must not be nil')

	local selfSceneNode = selfNode:getSceneNode()
	local childSceneNode = childNode:getSceneNode()

	context:getSceneNodeOpsQueue():queueOp(
		selfSceneNode,
		addOrRemoveChild,
		childSceneNode)
end

function DynamicLoaderView:getLoadedView()
	return self:getChildElement(self.loadedViewIndex)
end

function DynamicLoaderView:getRetainedObjectId()
	if self._loadedView ~= nil and self._loadedView.getRetainedObjectId then
		return self._loadedView:getRetainedObjectId()
	end

	return nil
end

return DynamicLoaderView
