-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local RetainedObjectUtils = plugins.views:lazyRequire('utils.RetainedObjectUtils')

local OPERATION_CREATE = 0
local OPERATION_UPDATE = 1
local OPERATION_DELETE = 2
local OPERATION_IGNORE = 3

local OpsContainer = class(function(self, correspondingModelObject)
	self._type = OPERATION_UPDATE
	self._localOps = {}
	self._childOps = {}
	self._aggregateOps = {}
	
	if correspondingModelObject ~= nil then
		self._model = correspondingModelObject
		self._objectId = RetainedObjectUtils.getObjectId(correspondingModelObject)
	end
end)

OpsContainer.OPERATION_CREATE = OPERATION_CREATE
OpsContainer.OPERATION_UPDATE = OPERATION_UPDATE
OpsContainer.OPERATION_DELETE = OPERATION_DELETE
OpsContainer.OPERATION_IGNORE = OPERATION_IGNORE

local function _getNames(table)
	local names = {}
	for name in pairs(table) do
		names[#names + 1] = name
	end
	return names
end

local function _countMembers(table)
	local count = 0
	for name in pairs(table) do
		count = count + 1
	end
	return count
end

local function _deepClone(object)
	if type(object) == 'table' then
		local clone = {}
		for k, v in pairs(object) do
			clone[k] = _deepClone(v)
		end
		return clone
	else
		return object
	end
end

function OpsContainer:getModel()
	return self._model
end

function OpsContainer:getObjectId()
	return self._objectId
end

function OpsContainer:getType()
	return self._type
end

function OpsContainer:setType(type)
	self._type = type

	-- Return the self object so that chained calls can be made.
	return self
end

function OpsContainer:hasLocal(name)
	return self._localOps[name] ~= nil
end

function OpsContainer:getLocal(name)
	return self._localOps[name]
end

function OpsContainer:getLocals()
	return self._localOps
end

function OpsContainer:getLocalNames()
	return _getNames(self._localOps)
end

function OpsContainer:getNumLocals()
	return _countMembers(self._localOps)
end

function OpsContainer:setLocal(name, value, type)
	self._localOps[name] =
	{
		type = type and type or OPERATION_CREATE,
		value = value
	}
end

function OpsContainer:extendLocal(name, propertyName, value)
	self._localOps[name] = self._localOps[name] or {}
	self._localOps[name][propertyName] = value
end

function OpsContainer:cloneToLocal(destName, sourceOp)
	self:setLocal(destName, sourceOp.value, sourceOp.type)
	self:extendLocal(destName, "oldValue", sourceOp.oldValue)
end

function OpsContainer:hasChild(name)
	return self._childOps[name] ~= nil
end

function OpsContainer:getChild(name)
	return self._childOps[name]
end

function OpsContainer:getChildren()
	return self._childOps
end

function OpsContainer:getChildNames()
	return _getNames(self._childOps)
end

function OpsContainer:getNumChildren()
	return _countMembers(self._childOps)
end

function OpsContainer:addChild(name, correspondingModelObject)
	local child = self._childOps[name]

	if child == nil then
		child = OpsContainer.new(correspondingModelObject)
		self._childOps[name] = child
	end

	return child
end

function OpsContainer:cloneToChild(destName, sourceOp)
	self._childOps[destName] = sourceOp
end

function OpsContainer:hasAggregate(index)
	return self._aggregateOps[index] ~= nil
end

function OpsContainer:getAggregate(index)
	return self._aggregateOps[index]
end

function OpsContainer:getAggregateIndices()
	return _getNames(self._aggregateOps)
end

function OpsContainer:getNumAggregates()
	return _countMembers(self._aggregateOps)
end

function OpsContainer:addAggregate(index)
	local aggregate = self._aggregateOps[index]

	if aggregate == nil then
		aggregate = OpsContainer.new()
		self._aggregateOps[index] = aggregate
	end

	return aggregate
end

function OpsContainer:cloneToAggregate(destIndex, sourceOp)
	self._aggregateOps[destIndex] = sourceOp
end

function OpsContainer:clearAggregates()
	self._aggregateOps = {}
end

function _typeToString(type)
	if type == OpsContainer.OPERATION_CREATE then
		return "OPERATION_CREATE"
	elseif type == OpsContainer.OPERATION_UPDATE then
		return "OPERATION_UPDATE"
	elseif type == OpsContainer.OPERATION_DELETE then
		return "OPERATION_DELETE"
	elseif type == OpsContainer.OPERATION_IGNORE then
		return "OPERATION_IGNORE"
	end

	return "UNKNOWN"
end

function OpsContainer:print(ident)
	ident = ident or 0
	local identStr = string.rep(" ", ident)

	for name, ops in pairs(self._localOps) do
		print(identStr .. _typeToString(ops.type) .. " (" .. name .. " => " .. tostring(ops.value) .. ")")
	end

	for name, ops in pairs(self._childOps) do
		print(identStr .. "** child " .. name .. ":")
		ops:print(ident+4)
	end

	for index, ops in pairs(self._aggregateOps) do
		print(identStr .. "** aggregate " .. tostring(index) .. ":")
		ops:print(ident+4)
	end
end

-- Map 'new' to 'create' in order to be hackwards compatible with OpsLuaBinding
OpsContainer.create = OpsContainer.new

return OpsContainer
