-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local ScrollingTimer = plugins.views:lazyRequire('utils.ScrollingTimer')
local IteratorOptions = plugins.views:lazyRequire('properties.iterator.IteratorOptions')

-- The ListScroller class abstracts the management of input handling, timing, and
-- when to activate and deactivate scrolling away from list views which require
-- scrolling behaviour. A client of the ListScroller must provide a 'scroll' callback
-- which actually scrolls the list itself, and also provide callbacks which provide
-- the ListScroller with knowledge about which direction to scroll in, and whether
-- there are elements available to scroll to. The client of the ListScroller can
-- pass a 'callbackContext' to the ListScroller which is then passed as the first
-- argument to each callback provided.
--
-- Example construction:
--
-- local listScroller = ListScroller({
--   keyRepeatInterceptor = KeyRepeatInterceptor.new(300),
--   scrollingTimer = ScrollingTimer.new(300),
--   callbackContext = self,
--   callbacks = {
--     scroll = function(context, direction, scrollingIteratorOptions) [scroll to next item] end
--     hasElementsInDirection = function(context, direction) [return boolean if can scroll in direction] end
--     getDirectionForKey = function(context, keyCode, keyModifier) [return "up", "down", "left" or "right"] end
--	   onStartScrolling = function(context, direction) [handle scrolling starting] end
--	   onStopScrolling = function(context, direction) [handle scrolling stopping] end
--   }
--})
--
-- NB. the 'onStartScrolling' and 'onStopScrolling' callbacks are optional

local ListScroller = class(function(self, options)
	self:_assertConstructorParamsAreValid(options)

	self._callbacks = {}
	self._callbackContext = options.callbackContext

	self._callbacks.scroll = options.callbacks.scroll
	self._callbacks.hasElementsInDirection = options.callbacks.hasElementsInDirection
	self._callbacks.getDirectionForKey = options.callbacks.getDirectionForKey
	self._callbacks.onStartScrolling = options.callbacks.onStartScrolling or function() end
	self._callbacks.onStopScrolling = options.callbacks.onStopScrolling or function() end

	self._scrollingTimer = options.scrollingTimer
	self._scrollingTimer:onInterval(self._scroll, self)

	self._inputInterceptor = options.keyRepeatInterceptor
	self._inputInterceptor:onKeyRepeat(self._keyRepeatHandler, self)
	self._inputInterceptor:onKeyUp(self._keyUpHandler, self)

	self._isScrolling = false
	self._scrollingDirection = nil
	self._scrollingIteratorOptions = IteratorOptions.new()
end)

function ListScroller:_assertConstructorParamsAreValid(options)
	assert(type(options.callbacks.scroll) == "function",
			"ListScroller scroll callback must be a function")
	assert(type(options.callbacks.hasElementsInDirection) == "function",
			"ListScroller hasElementsInDirection callback must be a function")
	assert(type(options.callbacks.getDirectionForKey) == "function",
			"ListScroller getDirectionForKey callback must be a function")
	assert(options.scrollingTimer ~= nil,
			 "The ScrollingTimer passed to ListScroller must not be nil")
	assert(options.keyRepeatInterceptor ~= nil,
			 "The KeyRepeatInterceptor passed to ListScroller must not be nil")
end

function ListScroller:activate()
	context:getTickableRegistry():addTickable(self._scrollingTimer)
	self._inputInterceptor:activate()
end

function ListScroller:deactivate()
	self._inputInterceptor:deactivate()
	if self:isScrolling() then
		scrollingDirection = self._scrollingDirection
		self._scrollingTimer:stop()
		context:getTickableRegistry():removeTickable(self._scrollingTimer)
		self._scrollingDirection = nil
		self._isScrolling = false
		self:_callCallback("onStopScrolling", scrollingDirection)
	end
end

function ListScroller:isScrolling()
	return self._isScrolling
end

function ListScroller:dispose()
	context:getTickableRegistry():removeTickable(self._scrollingTimer)
	self._inputInterceptor:dispose()
end

function ListScroller:setInterval(interval)
	self._scrollingTimer:setInterval(interval)
end

-- @param animationOptions.easingFunctionName
-- @param [animationOptions.easingFunctionParameters]
-- @param [animationOptions.duration]
function ListScroller:setAnimation(animationOptions)
	animation.loadFunction(animationOptions.easingFunctionName)

	self._scrollingIteratorOptions = IteratorOptions.new()
	self._scrollingIteratorOptions:addOption(
			"easingFn", animationOptions.easingFunctionName)
	if animationOptions.easingFunctionParameters ~= nil then
		self._scrollingIteratorOptions:addOption(
				"easingFnParams", animationOptions.easingFunctionParameters)
	end

	if animationOptions.duration ~= nil then
		self:setInterval(animationOptions.duration)
	end
end

function ListScroller:_scroll(interval)
	-- The scrolling iterator options passed in the scroll method, which will be populated
	-- if an animation was previously set on the listScroller, can be used to
	-- animate to the next position
	self._scrollingIteratorOptions:addOption("duration", self._scrollingTimer:getInterval())
	if self:_callCallback("hasElementsInDirection", self._scrollingDirection) then
		self:_callCallback("scroll",
			self._scrollingDirection,
			self._scrollingIteratorOptions)
	else
		self:deactivate()
	end
end

function ListScroller:_keyUpHandler(keyCode, keyModifier)
	local direction = self:_callCallback("getDirectionForKey", keyCode, keyModifier)
	if direction then
		self:deactivate()
		return false
	end
end

function ListScroller:_keyRepeatHandler(keyCode, keyModifier)
	local direction = self:_callCallback("getDirectionForKey", keyCode, keyModifier)
	if direction then
		self:_attemptToScrollInDirection(direction)
		return false
	end
end

function ListScroller:_attemptToScrollInDirection(direction)
	if not self:_callCallback("hasElementsInDirection", direction) then
		self:deactivate()
	elseif not self:isScrolling() then
		self._scrollingTimer:start(direction)
		self._scrollingDirection = direction
		self._isScrolling = true
		self:_callCallback("onStartScrolling", direction)
	end
end

function ListScroller:_callCallback(callbackName, ...)
	local callbackArgs = {...}
	local callbackFn = self._callbacks[callbackName]
	return callbackFn(self._callbackContext, unpack(callbackArgs))
end

return ListScroller
