-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local colors = plugins.views:lazyRequire("ansicolors.ansicolors")
local stringx = plugins.views:lazyRequire('pl.stringx')
local FileLoader = plugins.views:lazyRequire('utils.FileLoader')

local ErrorContextFinder = class()

-- Builds up a string describing the context of the XML file view was defined
-- in. Depending on how many arguments are supplied the string becomes more and
-- more detailed.
function ErrorContextFinder.attemptToRetrieveXmlContext(
		definitionScope,
		generatedLuaObject,
		erroneousStringToSearchFor)

	local context = ''

	-- If we've got an XML file path then start out by printing that, e.g.
	--
	--     "Defined in foo/bar/baz.xml"
	--
	if definitionScope ~= nil and definitionScope:getXmlFilePath() then
		context = '\n\nDefined in ' .. definitionScope:getXmlFilePath()

		-- If we were provided a Lua object that has an xmlLineNumber stored
		-- against it then append that, e.g.
		--
		--     "Defined in foo/bar/baz.xml at line 9"
		--
		if generatedLuaObject ~= nil and generatedLuaObject.xmlLineNumber ~= nil then
			context = context .. ' at line ' .. generatedLuaObject.xmlLineNumber

			-- The erroneousStringToSearchFor argument provides further context
			-- by letting us know which part of the XML line was bad - for example
			-- if the user tried to reference a fragment that doesn't exist. If
			-- we have this info then we can append a snippet of the XML file
			-- and underline the bit that's wrong, e.g.
			--
			--     Defined in foo/bar/baz.xml at line 9:
			--
			--          <fragment name="MistakenFragmentName"/>
			--                          ^^^^^^^^^^^^^^^^^^^^
			--
			if erroneousStringToSearchFor then
				local lineContents = ErrorContextFinder.findWithinFile(
					definitionScope:getXmlFilePath(),
					erroneousStringToSearchFor,
					generatedLuaObject.xmlLineNumber
				)

				if lineContents ~= nil then
					local indent = ''
					local underline = ''
					local position = lineContents:find(erroneousStringToSearchFor)

					for i = 1, position-1 do
						indent = indent .. ' '
					end

					for i = 1, #erroneousStringToSearchFor do
						underline = underline .. '^'
					end

					context =
						context ..
						':\n\n\t' .. lineContents ..
						'\n\t' .. indent .. colors('%{bright cyan}' .. underline)
				end
			end
		end
	end

	return context
end

-- Reads the contents of the supplied file and searches for the supplied
-- substring. If found, returns the contents of the line where string was found.
--
-- If the `startAtLine` argument is supplied, searches from that line forwards.
function ErrorContextFinder.findWithinFile(filePath, substring, startAtLine)

	-- It's important that ErrorContextFinder doesn't throw any errors itself,
	-- as this will only make things more confusing if we're in the middle of
	-- trying to print information of an error that's already taken place. As
	-- such it prints a warning and returns an empty string so that any
	-- concatenation of the return value will still work.
	if filePath == nil or substring == nil then
		log.warn('ErrorContextFinder.findWithinFile(): Required arguments missing')
		return ''
	end

	local source = FileLoader.loadFile(filePath)

	if source == nil then
		log.warn('ErrorContextFinder.findWithinFile(): Could not open ' .. filePath)
		return ''
	end

	local lines = stringx.splitlines(source)

	for lineNumber = startAtLine or 1, #lines do
		local lineContents = lines[lineNumber]

		if lineContents:find(substring) then
			return stringx.strip(lineContents)
		end
	end

	log.warn('ErrorContextFinder.findWithinFile(): Substring not found in file')
	return ''

end

return ErrorContextFinder