-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local WrappedPositionCalculator = class(function(
		self, elements, indexDelta, propertyManager, spacingCalculator)
		self._elements = elements
		assert(indexDelta ~= 0,	"WrappedPositionCalculator: " ..
				"Elements cannot be wrapped without a non-zero index delta")
		self._indexDelta = indexDelta
		self._propertyManager = propertyManager
		self._spacingCalculator = spacingCalculator

		self._referenceIndex = self:_getReferenceIndex()
		self._referencePosition = self:_getReferenceElementPosition()
end)

function WrappedPositionCalculator:_getReferenceIndex()
	-- The reference index is the index of the element which is at the end of
	-- the list of elements before wrapping has occurred; specifically the
	-- particular end of the list onto which we want to wrap elements which
	-- require wrapping
	if self._indexDelta < 0 then
		return #self._elements + self._indexDelta
	elseif self._indexDelta > 0 then
		return 1 + self._indexDelta
	end
end

function WrappedPositionCalculator:_getReferenceElementPosition()
	local referenceNode = self._elements[self._referenceIndex]:getRootNode():getSceneNode()
	return self._propertyManager:getValue(referenceNode)
end

function WrappedPositionCalculator:shouldWrapElement(elementIndex)
	if self._indexDelta < 0 then
		-- wrap element from beginning of list to the end of the list
		return elementIndex > self._referenceIndex
	elseif self._indexDelta > 0 then
		-- wrap element from end of the list to the beginning of the list
		return elementIndex < self._referenceIndex
	end
end

function WrappedPositionCalculator:getWrappedStartingPosition(elementIndex)
	return self._referencePosition + self:_calcReferenceElementOffset(elementIndex)
end

function WrappedPositionCalculator:_calcReferenceElementOffset(elementIndex)
	local lowIndex, highIndex, offsetMultiplier

	if self._referenceIndex > elementIndex then
		lowIndex = elementIndex
		highIndex = self._referenceIndex
		offsetMultiplier = -1
	else
		lowIndex = self._referenceIndex
		highIndex = elementIndex
		offsetMultiplier = 1
	end

	return self._spacingCalculator:getSpacingBetweenElements(
			lowIndex, highIndex) * offsetMultiplier
end

return WrappedPositionCalculator
