-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local IteratorOptions = plugins.views:lazyRequire("properties.iterator.IteratorOptions")
local IIterator = plugins.views:lazyRequire("properties.iterator.internal.IIterator")
local ViewPropertiesMixin = plugins.views:lazyRequire("utils.ViewPropertiesMixin")
local WrappedPositionCalculator = plugins.views:lazyRequire("properties.iterator.spacing.WrappedPositionCalculator")
local SpacingCalculator = plugins.views:lazyRequire("properties.iterator.spacing.SpacingCalculator")

local AbstractSpacingIterator = class(IIterator,
	function(self, propertyManager, multiplier)
		self:_checkParameter(
				propertyManager,
				"propertyManager",
				"table",
				"AbstractSpacingIterator")

		self:_checkParameter(
				multiplier,
				"multiplier",
				"number",
				"AbstractSpacingIterator")

		-- The multiplier variable helps adjusting the sign of the node size
		-- (width/height) so that it's possible to iterate in any direction
		-- (left/right, up/down)
		if multiplier ~= 1 and multiplier ~= -1 then
			error("The 'multiplier' argument must be 1 or -1.")
		end

		IIterator._ctor(self)

		self._multiplier = multiplier
		self._propertyManager = propertyManager
	end)


function AbstractSpacingIterator:_checkParameter(
		parameterValue, parameterName, expectedType, className)
	if (parameterValue == nil) or (type(parameterValue) ~= expectedType) then
		error(string.format("The '%s' argument to %s() is required "
			.. "and must be a '%s'.", parameterName, className, expectedType))
	end
end

function AbstractSpacingIterator:getIteratingDirections()
	return self._propertyManager:getIteratingDirections()
end

function AbstractSpacingIterator:doPropertyChangesAffectIteration(propertyNames)
	for _, propertyName in ipairs(propertyNames) do
		if self._propertyManager:getPropertiesAffectingNodeSize()[propertyName] then
			return true
		end
	end
	return false
end

function AbstractSpacingIterator:isIteratingDirection(direction)
	local iteratingDirections = self._propertyManager:getIteratingDirections()
	if iteratingDirections then
		return (iteratingDirections[direction] == true)
	end

	return false
end

function AbstractSpacingIterator:getPropertyManager()
	return self._propertyManager
end

function AbstractSpacingIterator:isForward(direction)
	if self._multiplier > 0 then
		return (direction=="right") or (direction=="up")
	else
		return (direction=="left") or (direction=="down")
	end
end

function AbstractSpacingIterator:isBackward(direction)
	if self._multiplier > 0 then
		return (direction=="left") or (direction=="down")
	else
		return (direction=="right") or (direction=="up")
	end
end

function AbstractSpacingIterator:iterate(elements, iteratorOptions)
	if elements and #elements > 0 then
		local iteratorOptions = iteratorOptions or IteratorOptions.new()
		self:_preprocess(elements, iteratorOptions)

		local easingFn = iteratorOptions.easingFn
		local easingFnParams = iteratorOptions.easingFnParams or {}
		local duration = iteratorOptions.duration
		local wrappingEnabled = iteratorOptions.wrappingEnabled
		local positionDelta = iteratorOptions.positionDelta
		local iterationRequiresWrapping = wrappingEnabled
			and positionDelta ~= 0
			and #elements > 1
		local wrappedPositionCalculator

		if iterationRequiresWrapping then
			wrappedPositionCalculator = self:_createWrappedPositionCalculator(
					elements, positionDelta)
		end

		local basePositions = self:_computeBasePositions(elements, iteratorOptions)
		local delta = self:_computeDelta(elements, iteratorOptions)
		delta = self:_adjustDelta(basePositions, delta, iteratorOptions)

		for i, elem in ipairs(elements) do
			local node = elem:getRootNode():getSceneNode()
			animation.clear(node, self._propertyManager:getPropertyName())

			local targetPos = basePositions[i] + delta
			local startingPos = nil

			if easingFn and duration and animation then
				if iterationRequiresWrapping then
					if wrappedPositionCalculator:shouldWrapElement(i) then
						startingPos = wrappedPositionCalculator:getWrappedStartingPosition(i)
					end
				end

				self:_queueAnimation(
					node, startingPos, targetPos, easingFn, easingFnParams, duration)
			else
				self._propertyManager:setValue(node, targetPos)
			end
		end

		self:_postprocess()
	end
end

function AbstractSpacingIterator:_createWrappedPositionCalculator(
		elements, positionDelta)
	local spacingCalculator = SpacingCalculator.new(
			elements, self._multiplier, self._propertyManager, self._margin)

	return WrappedPositionCalculator.new(
			elements, positionDelta, self._propertyManager, spacingCalculator)
end

function AbstractSpacingIterator:_preprocess(elements, iteratorOptions)
	self._extraSpacing = iteratorOptions.extraSpacing
	self._cumulativeSpacing = 0
end

function AbstractSpacingIterator:_postprocess()
	self._extraSpacing = nil
	self._cumulativeSpacing = nil
end

function AbstractSpacingIterator:_computeBasePositions(elements, iteratorOptions)
	local basePositions = {}

	if elements then
		for i, elem in ipairs(elements) do
			local node = elem:getRootNode():getSceneNode()

			local value = self:_iterationStep(i, node)
			value = value + self:_getExtraSpacingForIterationStep(i)
			table.insert(basePositions, value)
		end
	end

	return basePositions
end

function AbstractSpacingIterator:_getExtraSpacingForIterationStep(index)
	local extraSpacing = self._extraSpacing[index-1] or 0
	self._cumulativeSpacing = self._cumulativeSpacing + (self._multiplier * extraSpacing)
	return self._cumulativeSpacing
end

function AbstractSpacingIterator:_iterationStep(index, node)
	error("AbstractSpacingIterator:_iterationStep() is not implemented!")
end

function AbstractSpacingIterator:_computeDelta(elements, options)
	error("AbstractSpacingIterator:_computeDelta() is not implemented!")
end

function AbstractSpacingIterator:_adjustDelta(basePositions, delta, options)
	if options.firstPositionOffsetCap ~= nil then
		local pivotOffset = self._multiplier *
				(basePositions[options.itemInFirstPosition] + delta)

		if pivotOffset >= options.firstPositionOffsetCap then
			local deltaAdjustment = pivotOffset - options.firstPositionOffsetCap
			delta = delta - self._multiplier * deltaAdjustment
		end
	end

	return delta
end

function AbstractSpacingIterator:_queueAnimation(
		node, startValue, endValue, easingFn, easingFnParams, duration)
	local animationStart = ReflectableValueUserData.new()
	animationStart:set(startValue)
	local animationEnd = ReflectableValueUserData.new()
	animationEnd:set(endValue)

	animation.queue(
		node,
		self._propertyManager:getPropertyName(),
		animation.type.FIXED_DURATION,
		{
			from = animationStart,
			to = animationEnd,
			duration = duration,
			easingParams = easingFnParams,
			easing = easingFn,
			repetitions = 0
		})
end

function AbstractSpacingIterator:__eq(other)
	return self._multiplier == other._multiplier and
			self._propertyManager == other._propertyManager
end

return AbstractSpacingIterator
