-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local IteratorOptions = class(function(self, options)
	if (options ~= nil) and (type(options) ~= "table") then
		error("The 'options' argument to IteratorOptions must be a table")
	end

	self._addedOptions = {}

	for key, value in pairs(self._availableOptions) do
		self[key] = value.default
	end

	if options then
		for key, value in pairs(options) do
			self:addOption(key, value)
		end
	end
end)

function IteratorOptions:addOption(key, value)
	if self._availableOptions[key] then
		self[key] = value
		self._addedOptions[key] = value
	else
		error(string.format("'%s' is not a valid IteratorOption.", key))
	end
end

function IteratorOptions:merge(otherOptions)
	-- the options passed in to this function take precedence over this instance's
	-- current options
	for key, value in pairs(otherOptions._addedOptions) do
		self:addOption(key, value)
	end
end

IteratorOptions.typeName = "IteratorOptions"

IteratorOptions._availableOptions =
{
	-- [String] Easing function used to animate elements to target position
	easingFn =
	{
		default = nil
	},
	easingFnParams =
	{
		default = nil
	},
	-- [Number] Duration of the animation when moving elements to target position
	duration =
	{
		default = nil
	},
	-- [Map: Index -> Number] Extra spacing to be added to elements
	--
	-- E.g. Suppose elements would normally be moved to the
	--      following positions: [100, 200, 300]
	--
	-- If we provide "extraSpacing = {2 = 50}", all the elements after the
	-- second will be shifted by 50: [100, 200, 350]
	extraSpacing =
	{
		default = {}
	},
	-- [Index] Position in the grid where the "itemInFirstPosition" will
	--         be placed
	--
	-- E.g. Suppose we have a grid with the following positions: [100, 200, 300]
	--
	-- If we provide "startingPositionShift = 2", the first element will be moved
	-- to the "second position" in the grid above, and all the others will be
	-- shifted as a result. The final positions will be: [200, 300, 400]
	startingPositionShift =
	{
		default = 1
	},
	-- [Index] Index of the element that should occupy the first position in
	--         the grid
	--
	-- E.g. Suppose we have a grid with the following positions: [100, 200, 300]
	--
	-- If we provide "itemInFirstPosition = 2", the second element will be moved
	-- to the "first position" in the grid above, and all the others will be
	-- shifted as a result. The final positions will be: [-100, 100, 200]
	itemInFirstPosition =
	{
		default = 1
	},
	-- [Number] Maximum offset (from the origin of the list) in which the
	--          "itemInFirstPosition" can be positioned.
	--
	-- E.g. Suppose elements would normally be moved to the following positions:
	--      [100, 200, 300] and the "itemInFirstPosition" is 2
	--
	--      If the "firstPositionOffsetCap" is 50, the resulting positions
	--      will be: [-50, 50, 150]
	firstPositionOffsetCap =
	{
		default = nil
	},
	-- [Number] Number of items that were wrapped (only used if wrapping is
	--          enabled)
	positionDelta =
	{
		default = 0
	},
	-- [Boolean] Indicates if animations should be adjusted as to account for
	--           wrapping
	wrappingEnabled =
	{
		default = false
	}
}

return IteratorOptions
