-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local ParentDirective = DirectiveClass('NodeDirective')
local Directive = class(ParentDirective)

--[[
-- @inheritdoc
--
--]]
Directive.typeName = 'SceneNodeDirective'

function Directive:_createNode()
	return SceneNodeFactory.createSceneNode()
end

local function handleTag(node, method, value)
	-- Splits the 'value' string into a list of substrings, so we can pass
	-- multiple tags at once in view models, using space as a separator
	for tag in string.gmatch(value, "%S+") do
		method(node, tag)
	end
end

local function handleTagCreationOrDeletion(node, value, opType)
	if opType == OpsContainer.OPERATION_DELETE then
		if type(value) == 'table' then
			for _, tagName in ipairs(value) do
				handleTag(node, node.removeTag, tagName)
			end
		else
			handleTag(node, node.removeTag, value)
		end
	elseif opType == OpsContainer.OPERATION_CREATE then
		if type(value) == 'table' then
			for _, tagName in ipairs(value) do
				handleTag(node, node.addTag, tagName)
			end
		else
			handleTag(node, node.addTag, value)
		end
	end
end

local function tableToVec3(value)
	if type(value) == "table" then
		if (type(value.x) == "number") and (type(value.y) == "number") and
				(type(value.z) == "number") then
			return LuaVec3(value.x, value.y, value.z)
		end
	end

	return nil
end

Directive.setterTable =
{
	-- Position
	x = function (node, value) node:setX(value) end,
	y = function (node, value) node:setY(value) end,
	z = function (node, value) node:setZ(value) end,
	position = function(node, value)
		local vec = tableToVec3(value) or value
		node:setPosition(vec)
	end,

	-- Size
	width = function (node, value) node:setWidth(value) end,
	height = function (node, value) node:setHeight(value) end,
	depth = function (node, value) node:setDepth(value) end,
	size = function(node, value)
		local vec = tableToVec3(value) or value
		node:setSize(vec)
	end,

	alpha = function(node, value) node:setAlpha(value) end,
	drawIndex = function(node, value) node:setDrawIndex(value) end,

	-- Name
	name = function (node, value) node:setName(tostring(value)) end,

	-- View Name
	viewName = function (node, value) node:setViewName(value) end,

	-- Visibility
	visible = function (node, value) node:setVisible(value) end,

	tags = function (node, value, opType, oldValue)
		if opType == OpsContainer.OPERATION_UPDATE then
			handleTagCreationOrDeletion(node, oldValue, OpsContainer.OPERATION_DELETE)
			handleTagCreationOrDeletion(node, value, OpsContainer.OPERATION_CREATE)
		else
			handleTagCreationOrDeletion(node, value, opType)
		end
	end,

	-- Rotation
	rotationAxisX = function (node, value) node:setRotationAxisX(value) end,
	rotationAxisY = function (node, value) node:setRotationAxisY(value) end,
	rotationAxisZ = function (node, value) node:setRotationAxisZ(value) end,
	rotationAngle = function (node, value) node:setRotationAngle(value) end,

	-- Scale
	scaleX = function (node, value) node:setScaleX(value) end,
	scaleY = function (node, value) node:setScaleY(value) end,
	scaleZ = function (node, value) node:setScaleZ(value) end,
	scale = function(node, value)
		local vec = tableToVec3(value) or value
		node:setScale(vec)
	end
}


return Directive
