-- Copyright 2013 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local Directive = class(DirectiveClass())

--[[
-- @inheritdoc
--
--]]
Directive.typeName = 'NodeDirective'

--[[
-- Enumeration 'like' flag for marking setterTable setters as supporting
-- the PropertyAccessDecorator
--
--]]
Directive.USE_PROP_ACCESS_DECORATOR = true

--[[
-- Mapping of 'settable' properties to setter functions
--
--]]
Directive.setterTable = {}

--[[
-- Reference to the factory method used for creating and
-- returning Scene Node types
--
-- This value must be set by inheriting directives
-- @protected
--]]
function Directive:_createNode()
	error('Directive.createNode unimplemented. ' ..
			'This function must be overriden ')
end

--[[
-- @inheritdoc
--
--]]
function Directive:createResources()
	-- Create the one node this Directive supports
	self._rootNode = self:_createNode()
end

--[[
-- @inheritdoc
--
--]]
function Directive:setProperties(props)
	-- Check to see if the setter table as been defined
	-- The setterTable is required for this type of directive. It may
	-- however be empty.
	if type(self.setterTable) ~= 'table' then
		error('Directive.setterTable was missing. Must be defined as table')
	end

	-- Currently, this type of Directive only supports a single node
	-- We also unwrap the shared_ptr
	local node = self._rootNode:getSceneNode()

	-- Check to see if each prop has a local setter. If so, set
	for _, prop in pairs(props) do

		local propName = prop.name
		local propValue = prop.value

		local setter = self.setterTable[propName]
		if setter == nil then
--			log.warn('Attempt to set a value by propName', propName, 'without a valid setter function declared')
		else
			if (type(propValue) == 'table') and (propValue.type ~= nil) then
				setter(node, propValue.value, propValue.type, propValue.oldValue)
			else
				setter(node, propValue, OpsContainer.OPERATION_CREATE)
			end
		end
	end

	return true
end

--[[
-- @inheritdoc
--
--]]
function Directive:getRootNode()
	return self._rootNode
end

--[[
-- @inheritdoc
--
--]]
function Directive:dispose()
	if self:getRootNode() then
		context:getNodesRegistry():remove(
				self:getRootNode():getSceneNode():getId())
	end
	self._rootNode = nil
end

--[[
-- Utility method for merging setter tables.
-- Method takes the current table copies it to a new table and merges in the
-- new keys
--
--]]
function Directive:mergeSetters(partialSetters)
	local newSetters = {}

	local function mergeInto(subject)
		for key, value in pairs(subject) do
			newSetters[key] = value
		end
	end

	mergeInto(self.setterTable)
	mergeInto(partialSetters)

	return newSetters
end

return Directive
