-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local PreprocessFuncLib = {}

local lfs = lazyRequire("lfs")
local pl  = plugins.lua:lazyRequire("pl.path")

--[[
-- Contains a number of useful functions for use when dealing with views.
-- Functions are loaded from seperate lua files location in functions/ directory.
-- Any lua file placed in this directory will be loaded and it's return value
-- placed in to the function table, with the key of the table being the lua
-- filename.
--
--]]

local functionTable     = {}
local userFunctionPaths = {}

--[[
-- Adds a function to the internal functionTable.
-- Returns true if the addition was successful, or false if the function name
-- given was already present in the table.
--
--]]
local addFunction = function(functionName, func)
    if functionTable[functionName] then
        return false
    end

    functionTable[functionName] = func
    return true
end

--[[
--
--]]
local loadFunctionTable = function(absolutePath)
    for file in lfs.dir(absolutePath) do
        if file ~= "." and file ~= ".." then
            local funcName = string.match(file, "(.*).lua")
            if funcName then
                local func = lazyRequire(funcName)

                if type(func) ~= "function" then
                    error(tostring(file) .. " does not return a function!")
                else
                    addFunction(funcName, func)
                end
            end
        end
    end
end

--[[
-- Reloads the contents of the library table, loading the core built-in functions
-- and user-provided function paths.
--
--]]
local reloadFunctionTable = function()

    local setCoreFunctionPath = function(path)
        if not string.find(package.path, path) then
            package.path = package.path .. ";" .. path .. "/?.lua"
        end
    end

    local coreFunctionPath = pl.abspath(
            plugins.resolvePath('static://com.amazon.ignition.framework.views.logic/scripts/lua/core/utils/functions/'))

    -- Make sure core functions are set in package.path
    setCoreFunctionPath(coreFunctionPath)

    -- Empty the current table
    for funcKey in pairs(functionTable) do
        functionTable[funcKey] = nil
    end

    -- Core functions
    loadFunctionTable(coreFunctionPath)

    -- User-provided functions
    for pathKey in pairs(userFunctionPaths) do
        loadFunctionTable(userFunctionPaths[pathKey])
    end
end

--[[
-- Adds a given path to Lua's package.path, and also appends it to the 
-- user path table, so the contained files can be iterated and loaded.
--
--]]
function PreprocessFuncLib.addPath(path)
    -- Ensure the path is absolute
    path = pl.abspath(path)

    -- Append to the package path to allow us to load modules easily. Check the
    -- contents though, as we shouldn't 'overload' this path.
    if not string.find(package.path, path) then
        package.path = package.path .. ";" .. path .. "/?.lua"
    end
    userFunctionPaths[#userFunctionPaths+1] = path
    reloadFunctionTable()
end

--[[
-- Returns the given function from the functionTable. Checks if the table is 
-- empty and loads the functions if it is. Returns nil if the given function
-- doesn't exist.
--
--]]
function PreprocessFuncLib.getFunction(functionName)
    -- Lazy load the lua function files
    if next(functionTable) == nil then
        reloadFunctionTable()
    end
    return functionTable[functionName]
end

--[[
-- Adds a function to the internal functionTable.
-- Returns true if the addition was successful, or false if the function name
-- given was already present in the table.
--
--]]
function PreprocessFuncLib.addFunction(functionName, func)
    return addFunction(functionName, func)
end

return PreprocessFuncLib
