-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local MutationType = plugins.style:lazyRequire('mutations.MutationType')
local RuleApplicationStages = plugins.style:lazyRequire('rules.RuleApplicationStages')
local TableUtils = plugins.style:lazyRequire('utils.TableUtils')
local StyleValueFormatter = plugins.style:lazyRequire('utils.StyleValueFormatter')

local ffi = require('ffi')

local PrintableNodeMutationInfo = class(function(
		self, 
		sceneNode, 
		mutations, 
		progressFlags, 
		bindingUpdates,
		styleApplications, 
		matchesForNode,
		ruleSet)
		
	self._sceneNode = sceneNode
	self._mutations = mutations
	self._progressFlags = progressFlags
	self._bindingUpdates = bindingUpdates
	self._styleApplications = styleApplications
	self._matchesForNode = matchesForNode
	self._ruleSet = ruleSet
	
end)

function PrintableNodeMutationInfo:getSceneNode()
	return self._sceneNode
end

function PrintableNodeMutationInfo:getMutations()
	return self._mutations
end

function PrintableNodeMutationInfo:getProgressFlags()
	return self._progressFlags
end

function PrintableNodeMutationInfo:getBindingUpdates()
	return self._bindingUpdates
end

function PrintableNodeMutationInfo:getStyleApplications()
	return self._styleApplications
end

local SMALL_FONT = 12
local LARGE_FONT = 16

local _ruleApplicationStagesToColors = 
{
	[RuleApplicationStages.MATCHABILITY_CACHE_CHECK] = 'blue',
	[RuleApplicationStages.MATCHING_ALGORITHM] = 'orange',
	[RuleApplicationStages.STYLE_DELTA_PROCESSING] = 'red',
	[RuleApplicationStages.BINDING_RELINKING] = 'green',
}

function PrintableNodeMutationInfo:getGraphNodeString()
	local id = self._sceneNode:getId()
	local style
	local color
	
	-- If the node has been mutated, render it with a filled rectangle and
	-- choose its color based on the progress it made through the matching
	-- algorithm.
	if self._mutations ~= nil then
		style = 'filled'
		color = self:_chooseColorBasedOnProgressFlags()
		
	-- If the node wasn't mutated, just give it a solid black border and fill
	-- it white.
	else
		style = 'solid'
		color = 'white'	
	end
	
	return id .. ' [' .. 
		' shape=box' .. 
		' style=' .. style .. 
		' fillcolor=' .. color .. 
		' label=' .. self:_buildTableHtml() ..
		' ];'
end

function PrintableNodeMutationInfo:_chooseColorBasedOnProgressFlags()
	local color
	
	-- The colours below are intended to give a rough visual feel of the cost of 
	-- the final stage that each node reached in the rule application algorithm.
	--
	-- Reaching the style delta processing stage is the most expensive thing
	-- that can happen, so gets colored red.
	if self:_hasProgressFlag(RuleApplicationStages.STYLE_DELTA_PROCESSING) then
		color = 'red'		
	-- Hitting the matching algorithm is also expensive, so gets colored orange.
	elseif self:_hasProgressFlag(RuleApplicationStages.MATCHING_ALGORITHM) then
		color = 'orange'	
	-- The cheapest path is if we just skip the matching algorithm altogether 
	-- (i.e. if we get a match cache hit) and so just end up relinking bindings.
	elseif self:_hasProgressFlag(RuleApplicationStages.BINDING_RELINKING) then
		color = 'green'
	end
	
	if color == nil then
		local nodeId = self._sceneNode:getId()
		
		log.error(
			'Could not determine a printable color for node ' .. nodeId .. '. ' ..
			'Progress flags were: ' .. table.concat(self._progressFlags, ', '))
			
		return 'white'
	else
		return color
	end
end

function PrintableNodeMutationInfo:_hasProgressFlag(flag)
	return TableUtils.arrayContainsValue(self._progressFlags, flag)
end

function PrintableNodeMutationInfo:_buildTableHtml()
	local node = self._sceneNode
	local html = {}
	
	-- Open the container table
	table.insert(html, '<<table border="0">')
	
	-- Basic node info - id, type and name
	self:_pushRow(html, node:getId(), LARGE_FONT)
	self:_pushRow(html, ffi.string(node:getType()), LARGE_FONT)
	self:_pushRow(html, ffi.string(node:getName()), LARGE_FONT)
	self:_pushRow(html, ffi.string(node:getViewName()), LARGE_FONT)
	self:_pushRow(html, node:getMatchabilityKey(), LARGE_FONT)
	
	-- List of mutations
	table.insert(html, self:_buildMutationsTableHtml())
	
	-- List of tags
	table.insert(html, self:_buildTagsTableHtml())
	
	-- List of matching rule selectors
	table.insert(html, self:_buildMatchesTableHtml())
	
	-- Description of progress through the rule application algorithm
	table.insert(html, self:_buildProgressTableHtml())
	
	-- List of style applications
	table.insert(html, self:_buildStyleApplicationsTableHtml())

	-- Close the container table
	table.insert(html, '</table>>')
	
	return table.concat(html, '')
end

function PrintableNodeMutationInfo:_pushRow(html, label, size, align, class)
	if label ~= nil and label ~= '' then
		table.insert(html,  
			'<tr>' .. 
				'<td align="' .. (align or 'center') .. '">' ..
					'<font point-size="' .. size .. '">' .. label .. '</font>' ..
				'</td>' .. 
			'</tr>')
	end
end

function PrintableNodeMutationInfo:_buildMutationsTableHtml()
	return self:_buildTable('Mutations:', self._mutations, function(mutation)
		return self:formatMutationString(mutation)
	end)
end

function PrintableNodeMutationInfo:formatMutationString(mutation)
	local name
	local metadata = {}

	-- If the mutation value is a table, it is a complex mutation type
	-- which also has associated metadata (tag additions and removals
	-- are examples of this, where the name of the tag is included as
	-- metadata).
	if type(mutation) == 'table' then
		name = MutationType.getName(mutation.mutationType)

		for key, value in pairs(mutation) do
			if key ~= 'mutationType' then
				table.insert(metadata, self:_formatMutationMetadataPair(key, value)) 
			end
		end
	-- Otherwise it's a simple mutation, and the mutation value is just 
	-- the corresponding MutationType enum value.
	else
		name = MutationType.getName(mutation)
	end
	
	if #metadata > 0 then
		return name .. ' { ' .. table.concat(metadata, ', ') .. ' }'
	else
		return name
	end
end

function PrintableNodeMutationInfo:_formatMutationMetadataPair(key, value)
	if key == 'tag' then
		-- TAG_ADDED and TAG_REMOVED mutations include the tag's hash in their
		-- metadata, so we use the getTagNameForTagHash() method to translate
		-- it back into the actual tag before it gets printed.
		--
		-- In cases where a tag was added which isn't actually mentioned anywhere
		-- in a stylesheet, it's possible for the value to be unknown - in this
		-- case we print the hash so that it can still be manually cross referenced
		-- against the original tag if required.
		value = self._ruleSet:getTagNameForTagHash(value) or 'UNKNOWN (' .. value .. ')'
	end
	
	return key .. ': ' .. tostring(value)
end

function PrintableNodeMutationInfo:_buildTagsTableHtml()
	
	local numTags = self._sceneNode:numberOfTags()
	
	if numTags > 0 then
		local tagsArray = ffi.new('const char*[' .. numTags .. ']', {})
		self._sceneNode:getTags(tagsArray)
		local tagsTable = {}
		
		for i = 0, numTags - 1 do
			table.insert(tagsTable, ffi.string(tagsArray[i]))
		end
		
		return self:_buildTable('Tags:', tagsTable)
	else
		return ''
	end
end

function PrintableNodeMutationInfo:_buildMatchesTableHtml()
	return self:_buildTable('Matches:', self._matchesForNode, function(match)
		return match.matchedExpression.rawString:gsub('>', '&gt;')
	end)
end

function PrintableNodeMutationInfo:_buildProgressTableHtml()
	return self:_buildTable('Progress:', self._progressFlags)
end

function PrintableNodeMutationInfo:_buildStyleApplicationsTableHtml()
	return self:_buildTable('Style Applications:', self._styleApplications, function(info)
		local propertyName = info.style:getName()
		local divide = 
			'</font></td>' .. 
			'<td align="left" class="style-value ' .. propertyName .. '"><font point-size="12">'
		return propertyName .. divide .. self:_sanitiseValue(info.value)
	end)
end

function PrintableNodeMutationInfo:_sanitiseValue(value)
	return StyleValueFormatter.formatValue(value):gsub('<', '&lt;'):gsub('>', '&gt;')
end

function PrintableNodeMutationInfo:_buildTable(title, rows, formatter)
	-- Default to a no-op formatter if none is supplied
	formatter = formatter or function(row) return row end

	if rows ~= nil and #rows > 0 then
		local html = {}
		
		self:_pushRow(html, title, SMALL_FONT, 'left')
		
		table.insert(html, '<tr><td><table border="0">')
		for i, row in ipairs(rows) do
			self:_pushRow(html, formatter(row), SMALL_FONT, 'left')
		end
		table.insert(html, '</table></td></tr>')
		
		return table.concat(html, '')
	else
		return ''
	end
end

function PrintableNodeMutationInfo:getParentageString()
	local parent = self._sceneNode:getParent()
	
	if parent ~= nil then
		local parentId = parent:getId()
		local selfId = self._sceneNode:getId()
	
		return parentId .. ' -> ' .. selfId
	else
		return nil
	end
end

return PrintableNodeMutationInfo