
local config = plugins.style:lazyRequire('selectors.config')
local ParseContext = plugins.style:lazyRequire('selectors.parsing.ParseContext')
local CombinatorLoader = plugins.style:lazyRequire('selectors.loaders.CombinatorLoader')
local SymbolLoader = plugins.style:lazyRequire('selectors.loaders.SymbolLoader')
local Selector = plugins.style:lazyRequire('selectors.Selector')

local _parseExpressions
local _haveReachedNextExpression
local _haveReachedNextSegment
local _haveReachedStartOfNodeType
local _haveReachedStartOfTagName
local _nextExpression
local _nextSegment
local _retrieveRawStringForCurrentExpression
local _retrieveRawStringForCurrentSegment
local _readCombinationDefinition
local _readSymbolContent
local _calculateExpressionSpecificities
local _printParseError
local _generateStringOfLength
local _loadCombinators
local _combinatorDefinitions
local _loadSymbols
local _symbolDefinitions
local _defaultCombinatorDefinition

local ParsingEngine = class(function(self)

	_loadCombinators()
	_loadSymbols()

end)

local errors =
{
	UNEXPECTED_TOKEN = 'Unexpected token found in selector at index %i near %q',
	INVALID_COMBINATOR_DEFINITION = 'Error parsing combinator definition %q: ',
	INVALID_SYMBOL_DEFINITION = 'Error parsing symbol definition %q: '
}

local patterns =
{
	-- Delimits distinct expressions in selector strings, e.g "ImageNode, VideoNode"
	EXPRESSION_DELIMITER = ',',

	-- Delimits segments of an expression, e.g. the '>' in "ImageNode.foo > .bar"
	-- Note that this is nil initially, but is then populated automatically when
	-- the combination functions are loaded
	SEGMENT_DELIMITER = nil,

	-- Identifies the start of an individual symbol, e.g. "ImageNode" or ".foo".
	-- Again, these is nil initially, but is populated when the symbols are loaded.
	SYMBOL_PREFIX = nil,
}

function ParsingEngine:parse(selectorString)
	if ((selectorString == nil) or (selectorString == '')) then
		return {}
	else
		return _parseExpressions(ParseContext.new(selectorString))
	end
end

function _parseExpressions(context)
	local expression = _nextExpression(context)
	local segment = _nextSegment(context)

	while context:isAtEnd() == false do

		-- Check if we've reached an expression separator, i.e. ','
		if context:match(patterns.EXPRESSION_DELIMITER) then
			_retrieveRawStringForCurrentExpression(context)
			_retrieveRawStringForCurrentSegment(context)

			expression = _nextExpression(context)
			segment = _nextSegment(context)
			context:seek()           -- skip past the ',' character
			context:skipWhitespace() -- skip any leading whitespace
			context:flagExpressionStarted()
			context:flagSegmentStarted()

		-- Check if we've reached a segment separator, e.g. ' ', '>', etc
		elseif context:match(patterns.SEGMENT_DELIMITER) then
			_retrieveRawStringForCurrentSegment(context)

			segment = _nextSegment(context)
			context:flagSegmentStarted()

		-- Check whether the next character is the start of a symbol, e.g. '.'
		elseif context:match(patterns.SYMBOL_PREFIX) then
			_readSymbolContent(segment, context)

		-- If all of the above fail, we must not recognise the character
		else
			_printParseError(context, errors.UNEXPECTED_TOKEN)

		end

	end

	-- Extract the raw string making up the final expression and segment
	_retrieveRawStringForCurrentExpression(context)
	_retrieveRawStringForCurrentSegment(context)

	-- Calculate the specificity rating of each expression
	_calculateExpressionSpecificities(context:getExpressions())

	return Selector.new(context:getSelectorString(), context:getExpressions())
end

function _nextExpression(context)
	return context:addExpression()
end

function _nextSegment(context)
	local combinationDefinition = _readCombinationDefinition(context)
	local combinationMode = combinationDefinition.name

	return context:getCurrentExpression():addSegment(combinationMode)
end

function _retrieveRawStringForCurrentExpression(context)
	local currentExpression = context:getCurrentExpression()
	local expressionStart   = context:getExpressionStartIndex()
	local expressionEnd     = context:getIndex() - 1

	currentExpression.rawString = context:getSubstring(expressionStart, expressionEnd)
end

function _retrieveRawStringForCurrentSegment(context)
	local currentExpression = context:getCurrentExpression()
	local currentSegment    = currentExpression.segments[#currentExpression.segments]

	if currentSegment then
		local segmentStart = context:getSegmentStartIndex()
		local segmentEnd   = context:getIndex() - 1

		currentSegment.rawString = context:getSubstring(segmentStart, segmentEnd)
	end
end

function _readCombinationDefinition(context)
	local combinatorDefinition = _defaultCombinatorDefinition

	context:skipWhitespace()

	-- Loop over each of the combinator definitions we've loaded, checking whether
	-- its combinator character matches the character we've currently reached.
	--
	-- If so, we return the definition of the one we've matched.
	for i, definition in ipairs(_combinatorDefinitions) do

		if context:match(definition.character) then
			combinatorDefinition = definition
			context:seek() -- Skip past the combinator character, e.g. '>'
			break
		end

	end

	context:skipWhitespace()

	return combinatorDefinition
end

function _readSymbolContent(segment, context)
	-- Loop over each of the symbol definitions we've loaded, checking whether its
	-- pattern matches the character the context has currently read up to.
	for i, symbolDefinition in ipairs(_symbolDefinitions) do

		-- If we get a match, ask the definition's reader() method to read from the
		-- context until the end of the symbol.
		if context:match(symbolDefinition.prefix) then
			symbolDefinition.reader(segment, context)
			break
		end

	end
end

function _calculateExpressionSpecificities(expressions)
	-- Ask each of the expressions we've parsed to calculate its own specificity
	for i, expression in ipairs(expressions) do
		expression:calculateSpecificity()
	end
end

function _printParseError(context, message)
	local index = context:getIndex()
	local selector = context:getSelectorString()
	local character = context:getCurrentChar()

	-- Start with the formatted error message
	local message = message:format(index, character)

	-- Add a helpful line to show where the error is
	local indicatorArrow = _generateStringOfLength(index-1, ' ') .. '^'

	error(message .. '\n\n' .. selector .. '\n' .. indicatorArrow)
end

function _generateStringOfLength(length, char)
	local chars = ''

	while (#chars < length) do
		chars = chars .. char
	end

	return chars
end

function _loadCombinators()
	_combinatorDefinitions = CombinatorLoader.getCombinatorsAsList()

	local segmentDelimiters = ''
	for i, definition in ipairs(_combinatorDefinitions) do
		segmentDelimiters = segmentDelimiters .. definition.character

		-- the default is descendant if no other character matches
		if definition.name == config.combinationModes.DESCENDANT then
			_defaultCombinatorDefinition = definition
		end

	end

	patterns.SEGMENT_DELIMITER = '[' .. segmentDelimiters .. ']'
end

function _loadSymbols()
	_symbolDefinitions = SymbolLoader.getSymbolsAsList()

	local symbolPrefixes = ''
	for i, definition in ipairs(_symbolDefinitions) do
		symbolPrefixes = symbolPrefixes .. definition.prefix
	end

	patterns.SYMBOL_PREFIX = '[' .. symbolPrefixes .. ']'
end

return ParsingEngine
