
local TableUtils = plugins.style:lazyRequire('utils.TableUtils')

--
-- Maintains a map of node IDs to rules which are currently applied to them.
--
-- This is updated each time MatchingEngine generates a new set of matches for
-- a given node - old matches which are no longer relevant are removed, and new
-- matches are added.
--
local MatchRegistry = class(function(self)

	self._nodeIdsToMatches = {}
	self._matchabilityKeysToMatches = {}

end)

-- Returns an array of all current matches for the supplied node.
--
-- If the supplied node does not have any current matches, returns nil.
function MatchRegistry:getMatchesForNode(node)

	return self._nodeIdsToMatches[node:getId()]

end

-- Adds the supplied list of matches to the registry against the given node.
--
-- If any matches have changed, returns true. Otherwise returns false.
function MatchRegistry:setMatchesForNode(node, matches)

	local nodeId = node:getId()
	local previousMatches = self._nodeIdsToMatches[nodeId]

	self._nodeIdsToMatches[nodeId] = matches

	return self:_checkIfMatchesHaveChanged(matches, previousMatches)

end

function MatchRegistry:_checkIfMatchesHaveChanged(currentMatches, previousMatches)

	if previousMatches == nil
	or #currentMatches ~= #previousMatches then 
		return true
	else
		for i, match in pairs(currentMatches) do
			if match.rule ~= previousMatches[i].rule then
				return true
			end
		end
	end
	
	return false

end

-- Removes any matches registered against the supplied node ID.
function MatchRegistry:removeMatchesForNodeId(nodeId)

	self._nodeIdsToMatches[nodeId] = nil

end

return MatchRegistry