-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local PropertyDefaults = plugins.style:lazyRequire('rules.styles.definitions.PropertyDefaults')

local _metatable = {}
local _lookupTable = {}

--
-- Null Object (http://en.wikipedia.org/wiki/Null_Object_pattern) used when
-- a node keyword (i.e. `previous` or `parent`) returns null. Responds to any 
-- calls to methods like `getX()`, `getWidth()` etc with the corresponding
-- value from the PropertyDefaults list, so that valid values will always be
-- available even though technically a null reference was returned.
--
local NullNode = class(function(self)
	setmetatable(self, _metatable)
end)

-- Create an __index metamethod which will look up each property name in
-- the getter lookup table and return it if present.
_metatable.__index = function(self, getterName)
	local getter = _lookupTable[getterName]

	if getter == nil then
		error(
			'No value exists in PropertyDefaults.lua for \'' .. getterName .. '\''
		)
	else
		return getter
	end
end

-- Build a lookup table mapping each getter name to a function which will
-- return its value.
for key, value in pairs(PropertyDefaults) do
	local getterName = 'get' .. key:sub(1, 1):upper() .. key:sub(2)
	
	_lookupTable[getterName] = function()
		return value 
	end
end

-- Static function for checking whether the supplied node is an instance of
-- NullNode, as opposed to being an actual node.
function NullNode.isNullNodeInstance(node)
	return getmetatable(node) == _metatable
end

return NullNode

