-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local PropertyStyle = plugins.style:lazyRequire('rules.styles.definitions.PropertyStyle')
local AnimationTrigger = plugins.style:lazyRequire('expressions.definitions.animations.AnimationTrigger')
local PropertyTypes = plugins.style:lazyRequire('rules.styles.definitions.PropertyTypes')

--
-- Created for each property belonging to an effect instance, e.g. for the Alpha
-- effect an EffectEffectPropertyStyle will be created to manage the application of
-- its 'u_alpha_value' property.
--
local EffectPropertyStyle = class(

	-- Parent class
	PropertyStyle,

	-- Constructor
	function(self, name, index, type, value, default, containingStyles, context)

		PropertyStyle._ctor(self, name, value, containingStyles, context)

		self._index = index
		self._type = type
		self._default = default

	end
)

function EffectPropertyStyle:getIndex()
	return self._index
end

function EffectPropertyStyle:getType()
	return self._type
end

function EffectPropertyStyle:getDefault()
	return self._default
end

local _effectPropertyTypesToSetterMethods =
{
	[PropertyTypes.INT]   = 'setEffectPropertyValueInt',
	[PropertyTypes.FLOAT] = 'setEffectPropertyValueFloat',
	[PropertyTypes.VEC2]  = 'setEffectPropertyValueVec2',
	[PropertyTypes.VEC3]  = 'setEffectPropertyValueVec3',
	[PropertyTypes.VEC4]  = 'setEffectPropertyValueVec4'
}

-- TODO (pcowburn) Having to pass the isFirstAdditionOfEffect flag through here
-- is dirty...would be better to have an _initPropertyValue method or similar
-- which does the job of setting using the decorator on the first addition.
function EffectPropertyStyle:_setPropertyValue(node, value, isFirstAdditionOfEffect)

	-- If the evaluation returned an animation trigger, instruct it to start.
	if AnimationTrigger.isAnimationTrigger(value) then
		-- Flag that the value is an animation, so that we know to clear the
		-- animation later when removeFromNode() is called.
		self._valueIsAnimation = true

		-- For the animation plugin to be able to set the value via SceneNode's
		-- reflective setPropertyValue() method, it needs to have been set
		-- at least once already so that the reflectable type is known. This
		-- call facilitates that.
		if isFirstAdditionOfEffect then
			self:_setUsingEffectPropertySetterMethod(node, self._default)
		end

		-- Start the animation
		value:trigger()

	-- Otherwise, apply the value via the property access decorator.
	else
		self:_setUsingEffectPropertySetterMethod(node, value)

	end

end

function EffectPropertyStyle:_setUsingEffectPropertySetterMethod(node, value)

	local setterName = _effectPropertyTypesToSetterMethods[self._type]
	local setterFunc = node[setterName]

	if setterFunc ~= nil then
		setterFunc(node, self._index, value)
	elseif setterName ~= nil then
		error('Could not find a setter function called ' .. setterName)
	else
		error('Could not find a setter function for property type ' .. self._type)
	end

end

return EffectPropertyStyle



