
-- Rules
local RuleApplier = plugins.style:lazyRequire('rules.RuleApplier')
local RuleSet = plugins.style:lazyRequire('rules.RuleSet')

-- Matching
local MatchingEngine = plugins.style:lazyRequire('selectors.matching.MatchingEngine')
local MatchSorter = plugins.style:lazyRequire('selectors.matching.MatchSorter')
local MatchRegistry = plugins.style:lazyRequire('selectors.matching.MatchRegistry')
local MatchabilityCache = plugins.style:lazyRequire('selectors.matching.MatchabilityCache')
local MatchabilityCacheInvalidator = plugins.style:lazyRequire('selectors.matching.MatchabilityCacheInvalidator')

-- Styles
local StyleRegistry = plugins.style:lazyRequire('rules.styles.StyleRegistry')
local SignalQueue = plugins.style:lazyRequire('rules.styles.SignalQueue')

-- Expressions
local RuntimeExpressionTypes = plugins.style:lazyRequire('expressions.RuntimeExpressionTypes')
local ExpressionEvaluator = plugins.style:lazyRequire('expressions.ExpressionEvaluator')
local BindingManager = plugins.style:lazyRequire('expressions.bindings.BindingManager')
local Bridge = plugins.style:lazyRequire('expressions.bridge.Bridge')

-- Profiling
local StatsManager = plugins.style:lazyRequire('profiling.StatsManager')
local Counters = plugins.style:require('profiling.Counters')
local MutationTracker = plugins.style:lazyRequire('utils.mutationGraph.MutationTracker')
local MutationGraphPrinter = plugins.style:lazyRequire('utils.mutationGraph.MutationGraphPrinter')

--
-- Groups together the significant actors and state involved in the application
-- of styles/rules so that they may be retrieved by different parts of the system.
--
local StyleApplicationContext = class(function(self)

	self._tickIndex = -1
	self._isViewsSupportEnabled = self:_checkForPresenceOfViewsPlugin()

	-- Profiling
	self._statsManager = StatsManager.new()
	self:_defineCounters()

	-- Styles
	self._styleRegistry  = StyleRegistry.new()
	self._signalQueue    = SignalQueue.new()

	-- Matching
	self._matchRegistry  = MatchRegistry.new()
	self._matchabilityCache = MatchabilityCache.new()
	self._mutationTracker = MutationTracker.new(self._matchRegistry, self._styleRegistry)
	self._mutationGraphPrinter = MutationGraphPrinter.new()
	self._matchingEngine = MatchingEngine.new(self._matchabilityCache, self._mutationTracker)
	self._matchSorter    = MatchSorter.new()

	-- Expressions
	self._bridge = Bridge.new()
	self._bindingManager = BindingManager.new(self, self._mutationTracker)
	self._expressionEvaluator = ExpressionEvaluator.new(self)
	self:_loadRuntimeExpressionsTypes()

	-- Rules
	self._ruleApplier = RuleApplier.new(self)
	self._ruleSet     = RuleSet.new()
	
	self:_configureMatchabilityCacheInvalidation()

end)

function StyleApplicationContext:_checkForPresenceOfViewsPlugin()
	return (plugins.getPluginAlias('views') ~= nil)
end

function StyleApplicationContext:_defineCounters()
	local currentCategory

	local function _category(name)
		currentCategory = self._statsManager:defineCounterCategory(name)
	end

	local function _counter(description, suffix)
		return self._statsManager:defineCounter(currentCategory, description, suffix)
	end
	
	_category('General')
	Counters.TOTAL_NODES = _counter('Total nodes')
	Counters.TOTAL_RULES = _counter('Total rules')

	_category('Matchability Cache')
	Counters.MATCHABILITY_CACHE_HITS = _counter('Matchability cache hits', ' nodes')
	Counters.MATCHABILITY_CACHE_MISSES = _counter('Matchability cache misses', ' nodes')
	
	_category('Candidate Rule Selection')
	Counters.WORST_CASE_TOTAL_CANDIDATES = _counter('Worst case total candidate rules')
	Counters.CANDIDATES_FROM_GLOBAL_SCOPE = _counter('Potential candidates from global scope')
	Counters.CANDIDATES_FROM_VIEW_SCOPES = _counter('Potential candidates from view scopes')
	Counters.RULES_INCLUDED_FROM_NODE_TYPE_BUCKETS = _counter('Rules included from node type buckets')
	Counters.RULES_INCLUDED_FROM_RIGHTMOST_TAG_BUCKETS = _counter('Rules included from rightmost tag buckets')
	Counters.TOTAL_RULES_SELECTED_AS_CANDIDATES = _counter('Total rules selected as candidates')
	
	_category('Selector Matching')
	Counters.TOTAL_SELECTORS = _counter('Total selectors for all candidate rules')
	Counters.SELECTORS_ELIMINATED_BY_BLOOM_FILTER = _counter('Selectors eliminated by bloom filter')
	Counters.SELECTORS_ELIMINATED_BY_TAG_COUNT = _counter('Selectors eliminated by tag count')
	Counters.SELECTORS_ELIMINATED_BY_TAG_MISMATCH = _counter('Selectors eliminated by tag mismatch')
	Counters.SELECTORS_ELIMINATED_BY_TYPE_MISMATCH = _counter('Selectors eliminated by type mismatch')
	Counters.TOTAL_MATCHES = _counter('Total matches')
	
	_category('Match Change Tracking')
	Counters.NODES_FOR_WHICH_MATCHES_HAVENT_CHANGED = _counter('Cases where matches haven\'t changed', ' nodes')
	Counters.NODES_FOR_WHICH_MATCHES_HAVE_CHANGED = _counter('Cases where matches have changed', ' nodes')

	_category('Application')
	Counters.DELTAS_CALCULATED = _counter('Style deltas calculated')
	Counters.STYLES_REMOVED = _counter('Styles removed from nodes')
	Counters.PROPERTY_STYLES_APPLIED = _counter('Properties applied to nodes')
	Counters.EFFECT_STYLES_APPLIED = _counter('Effects applied to nodes')
	Counters.VIEW_PROPERTY_STYLES_APPLIED = _counter('View properties applied to nodes')

	_category('Bindings')
	Counters.STATIC_WATCHERS_ASSESSED = _counter('Static watchers assessed for changes')
	Counters.STATIC_WATCHERS_UPDATED = _counter('Static watchers updated')
	Counters.MUTATED_WATCHERS_ASSESSED = _counter('Mutated watchers assessed for changes')
	Counters.MUTATED_WATCHERS_UPDATED = _counter('Mutated watchers updated')
	Counters.BINDINGS_ASSESSED_FOR_RELINKING = _counter('Bindings assessed for relinking')
	Counters.BINDING_RELINKINGS_PERFOMED = _counter('Binding relinkings performed')
	Counters.BINDINGS_UPDATED_POST_RELINKING = _counter('Bindings updated post relinking')
end

function StyleApplicationContext:_loadRuntimeExpressionsTypes()
	for k, v in pairs(RuntimeExpressionTypes) do
		local expression = plugins.style:lazyRequire('expressions.definitions.' .. v .. 'RuntimeExpression')
		self._expressionEvaluator:loadExpressionType(expression)
	end
end

function StyleApplicationContext:_configureMatchabilityCacheInvalidation()
	self._matchabilityCacheInvalidator = MatchabilityCacheInvalidator.new(
			self._matchabilityCache, 
			self._ruleSet)
			
	self._matchabilityCacheInvalidator:startListening()
end

function StyleApplicationContext:getTickIndex()
	return self._tickIndex
end

function StyleApplicationContext:isViewsSupportEnabled()
	return self._isViewsSupportEnabled
end

function StyleApplicationContext:setTickIndex(tickIndex)
	self._tickIndex = tickIndex
end

function StyleApplicationContext:getRuleApplier()
	return self._ruleApplier
end

function StyleApplicationContext:getRuleSet()
	return self._ruleSet
end

function StyleApplicationContext:getMatchingEngine()
	return self._matchingEngine
end

function StyleApplicationContext:getMatchSorter()
	return self._matchSorter
end

function StyleApplicationContext:getMatchRegistry()
	return self._matchRegistry
end

function StyleApplicationContext:getMatchabilityCache()
	return self._matchabilityCache
end

function StyleApplicationContext:getStyleRegistry()
	return self._styleRegistry
end

function StyleApplicationContext:getSignalQueue()
	return self._signalQueue
end

function StyleApplicationContext:getExpressionEvaluator()
	return self._expressionEvaluator
end

function StyleApplicationContext:getBindingManager()
	return self._bindingManager
end

function StyleApplicationContext:getBridge()
	return self._bridge
end

function StyleApplicationContext:getStatsManager()
	return self._statsManager
end

function StyleApplicationContext:getMutationTracker()
	return self._mutationTracker
end

function StyleApplicationContext:getMutationGraphPrinter()
	return self._mutationGraphPrinter
end

return StyleApplicationContext