-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local PropertyRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.PropertyRuntimeExpression')
local NullNode = plugins.style:lazyRequire('rules.styles.definitions.NullNode')

local _createIndexedExpressionCreator

--
-- Base class for all CurrentNodeKeywords. This class provides IndexedExpressionCreator
-- which creates PropertyRuntimeExpressions whenever the node, parent, previous
-- etc keywords are indexed during rule definition.
--
-- For example:
--
--    to "ImageNode.square" apply
--    {
--        x = 100,
--        y = node.x   <<< y must be evaluated by reading the value of x at runtime
--    }
--
-- When the above rule is defined, the statement "node.x" has to be converted
-- to a PropertyRuntimeExpression so that it can be evaluated in the application
-- state when a node matches the rule.
--
local AbstractNodeKeyword = class(function(self, sourceNodeType, isLive)

	self._sourceNodeType = sourceNodeType
	self._isLive = isLive
	self._indexedExpressionCreator = _createIndexedExpressionCreator(self, isLive)

end)

local TABLE_TYPE = 'table'

-- Instance of NullNode used when the traverse() method returns nil.
local _nullNode = NullNode.new()

-- Performs a traversal from the currentNode to whichever node should be targeted
-- by the concrete node keyword which implements _doTraversal().
function AbstractNodeKeyword:traverse(currentNode)
	local result = self:_doTraversal(currentNode)
	
	-- If we didn't get a valid node back from the traverse() method, default to
	-- using an instance of NullNode. This reads its property values from the 
	-- PropertyDefaults table, so there will always be a valid value available.
	if result == nil then
		return _nullNode
	else
		return result
	end
end

-- Implement this method in your derived classes to return the node from which
-- style properties should be read at runtime.
function AbstractNodeKeyword:_doTraversal(currentNode)
	error(
		'AbstractNodeKeyword:_doTraversal() cannot be called directly, please' ..
		'implement it in your derived class to provide the traversal behaviour.'
	)
end

-- Returns the SOURCE_NODE_TYPE that will be targeted by this keyword.
function AbstractNodeKeyword:getSourceNodeType()
	return self._sourceNodeType
end

-- Returns whether this node keyword returns live PropertyRuntimeExpressions or
-- just static ones.
function AbstractNodeKeyword:isLive()
	return self._isLive
end

-- Returns an IndexedExpressionCreator which will create PropertyRuntimeExpressions
-- when indexed.
function AbstractNodeKeyword:getIndexedExpressionCreator()
	return self._indexedExpressionCreator
end

-- Used for converting to a table representation when transferring from the
-- definition state to the application state.
--
-- NOTE: This is a generic method intended to convert the default properties of
-- a NodeKeyword to table format when transferring between states. Note that if
-- your derived NodeKeyword type has more properties than the ones added to the
-- table below, you will need to override toTable() in order to export all of
-- the necessary properties.
function AbstractNodeKeyword:toTable()
	return
	{
		isKeywordTable = true,
		isLive = self._isLive,
		sourceNodeType = self._sourceNodeType
	}
end

-- Used for identifying whether the supplied table was generated by the toTable()
-- method of AbstractNodeKeyword.
function AbstractNodeKeyword.isKeywordTable(value)
	return type(value) == TABLE_TYPE and value.isKeywordTable
end

-- Creates a table which has the __index method wired up to create new instances
-- of PropertyRuntimeExpression when stylesheet authors do things like 'node.x',
-- 'parent.width' etc.
function _createIndexedExpressionCreator(nodeKeywordInstance, isLive)

	local table = {}

	setmetatable(table,
	{
		__index = function (self, key)
			return PropertyRuntimeExpression.new(key, nodeKeywordInstance, isLive)
		end
	})

	return table

end

return AbstractNodeKeyword
