-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')

--
-- Used in tandem with RuleTableImporter to manage the transfer of rules from 
-- the rule definition Lua state to the rule application Lua state.
--
local RuleTableExporter = class()

-- Note: For performance's sake this method will convert the incoming `rules`
-- argument to its table representation IN PLACE. If you want to preserve the
-- table then you'll have to do a deep copy on it before passing it to this method.
function RuleTableExporter:convertToTableRepresentation(
		rules,
		partitionMappings)
	local counts = self:_generateCounts(rules)
	local tagLookupTable = self:_generateTagLookupTable(rules)

	self:_convertToTablesRecursively(rules)

	local rulesContainer =
	{
		numSelectors      = counts.selectors,
		numExpressions    = counts.expressions,
		numSegments       = counts.segments,
		rules             = rules,
		partitionMappings = partitionMappings,
		tagLookupTable    = tagLookupTable
	}

	return rulesContainer
end

-- Checks whether any style values are RuntimeExpressions, and if so, converts
-- them to their table representation via RuntimeExpression:toTable() so that
-- they can be transferred to the definition state.
function RuleTableExporter:_convertToTablesRecursively(rules)

	for i, v in pairs(rules) do
		if AbstractRuntimeExpression.isExpression(v) then
			rules[i] = v:toTable()
		end

		if type(rules[i]) == 'table' then
			self:_convertToTablesRecursively(rules[i])
		end
	end

end

function RuleTableExporter:_generateCounts(rules)

	local counts =
	{
		selectors = #rules,
		expressions = 0,
		segments = 0
	}

	for i = 1, #rules do
		local rule = rules[i]
		counts.expressions = counts.expressions + #rule.selector.expressions

		for j = 1, #rule.selector.expressions do
			local expression = rule.selector.expressions[j]
			counts.segments = counts.segments + #expression.segments
		end
	end

	return counts

end

function RuleTableExporter:_generateTagLookupTable(rules)

	local lookupTable = {}

	-- This loop looks insane, but in practice there's usually only one 
	-- expression per selector and one or two tags per segment, so really
	-- it's more like two nested loops rather than four.
	for i, rule in ipairs(rules) do
		for j, expression in ipairs(rule.selector.expressions) do
			for k, segment in ipairs(expression.segments) do
				for l, tag in ipairs(segment.tags) do
					lookupTable[tag.hash] = tag.name
				end
			end
		end
	end

	return lookupTable

end

return RuleTableExporter



