-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local djb2Hash = crypto.djb2Hash

local RuleSetPartition = plugins.style:lazyRequire('rules.RuleSetPartition')
local Signal = plugins.lua:lazyRequire('signals.Signal')

--
-- List of rules used by RuleApplier at rule application time.
--
-- Maintains a mapping of rules against the stylesheet file they originated
-- from, and provides an API allowing these to be whitelisted against scene 
-- graph partitions.
--
local RuleSet = class(function(self)
	
	self.partitionModifiedSignal = Signal.new()
	
	self:clear()
	
end)

local DEFAULT_PARTITION_NAME = '__default__'
local DEFAULT_PARTITION_NAME_HASH = djb2Hash(DEFAULT_PARTITION_NAME)

-- Constant used for specifying that a stylesheet should be mapped against all
-- partitions, rather than just one. This is used when loadStylesheet() is called
-- without passing the optional `partition` argument, which results in the rules
-- from the stylesheet being global.
local ALL_PARTITIONS = '__all__'
local ALL_PARTITIONS_HASH = djb2Hash(ALL_PARTITIONS)

-- Constant used as the key to the _rulesByStylesheetPath table when rules are 
-- being inserted which did not originate from a stylesheet file (i.e. those that
-- are interpreted directly from an in-memory string).
local NO_STYLESHEET_PATH = '__noPath__'

function RuleSet:clear()
	self._partitionsByNameHash = {}
	self._rulesByStylesheetPath = {}
	self._globalRules = {}
	self._tagLookupTable = {}
	
	self._numRules = 0
	self._numSelectors = 0
	self._numExpressions = 0
	self._numSegments = 0
	
	self:_getOrCreatePartition(DEFAULT_PARTITION_NAME_HASH)
end

function RuleSet:_getOrCreatePartition(nameHash)
	local partition = self._partitionsByNameHash[nameHash]
	
	if not partition then
		partition = RuleSetPartition.new(nameHash)
		partition:addRules(self._globalRules)
		
		self._partitionsByNameHash[nameHash] = partition
	end
	
	return partition
end

function RuleSet:importRules(rulesContainer)
	self._numRules = self._numRules + #rulesContainer.rules
	self._numSelectors = self._numSelectors + rulesContainer.numSelectors
	self._numExpressions = self._numExpressions + rulesContainer.numExpressions
	self._numSegments = self._numSegments + rulesContainer.numSegments

	self:_addRulesToRelevantStylesheetRulesList(rulesContainer.rules)
	self:_processPartitionMappings(rulesContainer.partitionMappings)
	self:_addTagsToLookupTable(rulesContainer.tagLookupTable)
end

function RuleSet:_addRulesToRelevantStylesheetRulesList(rules)
	local newGlobalRules = {}

	for i, rule in ipairs(rules) do
		local path = rule.stylesheetPath
		
		-- Any rules which were not loaded from an actual stylesheet file (i.e. 
		-- those that have been loaded directly from an in-memory string) become
		-- global by default. Otherwise there would be no way to actually apply 
		-- them to any nodes, as partition mapping works on the basis of the 
		-- stylesheet the rules came from.
		if path == nil then
			table.insert(newGlobalRules, rule)
			path = NO_STYLESHEET_PATH
		end
			
		table.insert(self:_getOrCreateStylesheetRulesList(path), rule)
	end
	
	if #newGlobalRules > 0 then
		self:_updateGlobalRules(newGlobalRules)
	end
end

function RuleSet:_updateGlobalRules(newGlobalRules)
	for i, rule in ipairs(newGlobalRules) do
		table.insert(self._globalRules, rule)
	end
	
	for nameHash, partition in pairs(self._partitionsByNameHash) do
		self:_addRulesToPartitionAndDispatchModifiedSignal(partition, newGlobalRules)
	end
end

function RuleSet:_addRulesToPartitionAndDispatchModifiedSignal(partition, rules)
	partition:addRules(rules)
	
	self.partitionModifiedSignal:dispatch(partition)
end

function RuleSet:_getOrCreateStylesheetRulesList(path)
	local rulesList = self._rulesByStylesheetPath[path]
	
	if not rulesList then
		rulesList = {}
		self._rulesByStylesheetPath[path] = rulesList
	end
	
	return rulesList
end

function RuleSet:_processPartitionMappings(partitionMappings)
	for stylesheetPath, partitions in pairs(partitionMappings) do
		for i, partitionName in pairs(partitions) do
			if partitionName == ALL_PARTITIONS then
				self:makeStylesheetGlobal(stylesheetPath)
			else
				self:mapStylesheetToPartition(stylesheetPath, partitionName)
			end
		end
	end
end

function RuleSet:makeStylesheetGlobal(stylesheetPath)
	local rules = self:getRulesForStylesheet(stylesheetPath)
	
	-- Some stylesheet files might not actually contain any rules - as is the 
	-- case with constants files, for example.
	if rules ~= nil then
		self:_updateGlobalRules(rules)
	end
end

function RuleSet:mapStylesheetToPartition(stylesheetPath, partitionName)
	local rules = self:getRulesForStylesheet(stylesheetPath)
	local partition = self:_getOrCreatePartition(djb2Hash(partitionName))
	
	if rules ~= nil then
		self:_addRulesToPartitionAndDispatchModifiedSignal(partition, rules)
	end
end

function RuleSet:_addTagsToLookupTable(tagNamesByHash)
	for hash, name in pairs(tagNamesByHash) do
		self._tagLookupTable[hash] = name
	end
end

function RuleSet:getGlobalRules()
	return self._globalRules
end

function RuleSet:getRulesForStylesheet(stylesheetPath)
	return self._rulesByStylesheetPath[stylesheetPath]
end

function RuleSet:getPartitionByName(name)
	return self:getPartitionByNameHash(djb2Hash(name))
end

function RuleSet:getPartitionByNameHash(nameHash)
	return self:_getOrCreatePartition(nameHash)
end

function RuleSet:getTagNameForTagHash(tagHash)
	return self._tagLookupTable[tagHash]
end

function RuleSet:getAllRules()
	local allRules = {}
	
	for path, rules in pairs(self._rulesByStylesheetPath) do
		for i, rule in ipairs(rules) do
			table.insert(allRules, rule)
		end
	end
	
	return allRules
end

function RuleSet:getNumRules()
	return self._numRules
end

function RuleSet:getNumGlobalRules()
	return #self._globalRules
end

function RuleSet:getNumSelectors()
	return self._numSelectors
end

function RuleSet:getNumExpressions()
	return self._numExpressions
end

function RuleSet:getNumSegments()
	return self._numSegments
end

return RuleSet



