-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
--[[

	Interface designed for use in the Lua state that styles are defined within.

	Exposes the select, apply, get set methods, etc.

  ]]--

local function init()
	-- Set the plugin alias so we may convieniently load scripts usin:
	-- plugins.style:require(...)
	plugins.setPluginAlias('style', 'com.amazon.ignition.framework.style.logic');

	-- API
	local RuleDefiner = plugins.style:lazyRequire('rules.RuleDefiner')
	local RuleFinalizer = plugins.style:lazyRequire('rules.RuleFinalizer')
	local EffectLibrary = plugins.style:lazyRequire('effects.EffectLibrary')
	local EffectFactory = plugins.style:lazyRequire('effects.EffectFactory')
	local AnimationLibrary = plugins.style:lazyRequire('expressions.definitions.animations.AnimationLibrary')
	local IteratorsLibrary = plugins.style:lazyRequire('expressions.definitions.iterators.IteratorsLibrary')
	local AssetLibrary = plugins.style:lazyRequire('assets.AssetLibrary')
	local Bridge = plugins.style:lazyRequire('expressions.bridge.Bridge')
	local BridgeRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.BridgeRuntimeExpression')
	local AssetGetterFactory = plugins.style:lazyRequire('assets.AssetGetterFactory')
	local InputConstants = plugins.input:lazyRequire('external.inputConstants')
	local ZeroGuardRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.ZeroGuardRuntimeExpression')

	-- Keywords
	local BlendModeEnum = plugins.style:lazyRequire('rules.enums.BlendMode')
	local FontWeightEnum = plugins.style:lazyRequire('rules.enums.FontWeight')
	local FontStyleEnum = plugins.style:lazyRequire('rules.enums.FontStyle')
	local ParseModeEnum = plugins.style:lazyRequire('rules.enums.ParseMode')
	local TruncationModeEnum = plugins.style:lazyRequire('rules.enums.TruncationMode')
	local AlignmentEnum = plugins.style:lazyRequire('rules.enums.Alignment')
	local EasingEnum = plugins.style:lazyRequire('expressions.definitions.animations.Easing')
	local VerticalAlignmentEnum = plugins.style:lazyRequire('rules.enums.VerticalAlignment')
	local SourceChangeModeEnum = plugins.style:lazyRequire('rules.enums.SourceChangeMode')
	local ImageLoadModeEnum = plugins.style:lazyRequire('rules.enums.ImageLoadMode')
	local MipmappingModeEnum = plugins.style:lazyRequire('rules.enums.MipmappingMode')
	local RenderModeEnum = plugins.style:lazyRequire('rules.enums.RenderMode')
	local PivotModeEnum = plugins.style:lazyRequire('rules.enums.PivotMode')
	local SceneInteractivityMode = plugins.scene:lazyRequire('external.InteractivityMode')
	local PropertyTypesList = plugins.style:lazyRequire('rules.styles.definitions.PropertyTypes')
	local CurrentNodeKeyword = plugins.style:lazyRequire('rules.keywords.CurrentNodeKeyword')
	local ParentNodeKeyword = plugins.style:lazyRequire('rules.keywords.ParentNodeKeyword')
	local PreviousNodeKeyword = plugins.style:lazyRequire('rules.keywords.PreviousNodeKeyword')

	-- Types
	local Vec2Type = plugins.style:lazyRequire('expressions.definitions.types.Vec2Type')
	local Vec3Type = plugins.style:lazyRequire('expressions.definitions.types.Vec3Type')
	local Vec4Type = plugins.style:lazyRequire('expressions.definitions.types.Vec4Type')
	local RectType = plugins.style:lazyRequire('expressions.definitions.types.RectType')
	local PivotModeType = plugins.style:lazyRequire('expressions.definitions.types.PivotModeType')
	local GrayscaleColorType = plugins.style:lazyRequire('expressions.definitions.types.GrayscaleColorType')
	local RgbColorType = plugins.style:lazyRequire('expressions.definitions.types.RgbColorType')
	local HexColorType = plugins.style:lazyRequire('expressions.definitions.types.HexColorType')

	-- Private variables
	local _ruleDefiner
	local _assetLibrary
	local _bridge

	-- Creates each of the main actors that facilitate stylesheet definition
	local function _initEnvironment()
		local ruleFinalizer = RuleFinalizer.new()

		_ruleDefiner = RuleDefiner.new(ruleFinalizer)
		_assetLibrary = AssetLibrary.new()
		_bridge = Bridge.new()

		-- Give the EffectFactory a reference to the Bridge so that it can do
		-- type checking of Bridge variable arguments.
		EffectFactory.setBridge(_bridge)

		-- Accessor function intended for use only by the owner of the LuaState,
		-- allowing it to retrieve the RuleDefiner being used by the stylesheets
		__getRuleDefiner__ = function ()
			return _ruleDefiner
		end

		-- Same as previous function but for the AssetLibrary object
		__getAssetLibrary__ = function ()
			return _assetLibrary
		end

		-- Same as previous function but for the Bridge object
		__getBridge__ = function ()
			return _bridge
		end

	end

	-- Creates the keywords that stylesheet authors use to define styles
	local function _defineKeywords()

		-- Defines the 'to' keyword used to start a rule by specifying the
		-- selector that should be associated with it
		to = function (selector)
			_ruleDefiner:beginRule(selector)
		end

		-- Defines the 'apply' keyword used to specify the styles to associate
		-- with a selector
		apply = function (styles)
			_ruleDefiner:endRule(styles)
		end

		-- Adds an entry to the asset library using the supplied attributes.
		-- Returns the source of the asset, so that assets may be defined inline
		-- if desired.
		defineAsset = function (attributes)
			return _assetLibrary:defineAsset(attributes):getSource()
		end

		-- Returns the source of the asset with the supplied name.
		getAsset = function (name)
			return _assetLibrary:getAsset(name):getSource()
		end

		-- Registers the asset path for the given namespace
		registerAssetPath = function(namespace, path)
			 _assetLibrary:registerAssetPath(namespace, path)
		end

		-- Removes the asset path for the given namespace
		removeAssetPath = function(namespace)
			 _assetLibrary:removeAssetPath(namespace)
		end

		-- Returns the asset path registered for the given namespace
		getAssetPath = function(namespace)
			return _assetLibrary:getAssetPath(namespace)
		end

		-- Gets an asset for the given namespace (if it exists) or preload it
		-- otherwise (if it's not yet loaded).
		Assets = AssetGetterFactory.createGetterForAssetLibrary(_assetLibrary)

		-- Allows a variable name to be declared as a specific type
		-- and set it with a initial value of that type.
		declareVariable = function (name, type, initialValue)
			return _bridge:declareVariable(name, type, initialValue)
		end

		--Used to retrieve a variable value previously declared
		-- using 'declareVariable'
		getVariable = function (name)
			return BridgeRuntimeExpression.new(name)
		end

		-- Used to set a previously declared variable with a specific value.
		-- The value type should fit the one used during declaration of the
		-- variable when using 'declareVariable'
		setVariable = function (name, value)
			return _bridge:setVariable(name, value)
		end

		-- Used to protect against divide by zero errors. If the expression
		-- fails the guard check then the default value is returned.
		ZeroGuard = function (expression, defaultValue)
			return ZeroGuardRuntimeExpression.new(expression, defaultValue)
		end

		-- Defines the 'node' keyword used when referring to properties of
		-- the current node when writing styles.
		node = CurrentNodeKeyword.new():getIndexedExpressionCreator()

		-- The live counterpart of the node keyword (monitors for external
		-- changes to property values).
		liveNode = CurrentNodeKeyword.new(true):getIndexedExpressionCreator()

		-- Defines the 'parent' keyword used when referring to properties of
		-- the current node's parent when writing styles.
		parent = ParentNodeKeyword.new():getIndexedExpressionCreator()

		-- The live counterpart of the parent keyword (monitors for external
		-- changes to property values).
		liveParent = ParentNodeKeyword.new(true):getIndexedExpressionCreator()

		-- Defines the 'previous' keyword used when referring to properties of
		-- the current node's previous sibling when writing styles.
		previous = PreviousNodeKeyword.new():getIndexedExpressionCreator()

		-- The live counterpart of the previous keyword (monitors for external
		-- changes to property values).
		livePrevious = PreviousNodeKeyword.new(true):getIndexedExpressionCreator()

		-- Define various enums used in stylesheets
		BlendMode = BlendModeEnum
		FontWeight = FontWeightEnum
		FontStyle = FontStyleEnum
		ParseMode = ParseModeEnum
		TruncationMode = TruncationModeEnum
		Alignment = AlignmentEnum
		Easing = EasingEnum
		VerticalAlignment = VerticalAlignmentEnum
		SourceChangeMode = SourceChangeModeEnum
		ImageLoadMode = ImageLoadModeEnum
		MipmappingMode = MipmappingModeEnum
		RenderMode = RenderModeEnum
		Pivot = PivotModeEnum
		InteractivityFlag = SceneInteractivityMode.flagsEnum
		InteractivityMode = SceneInteractivityMode.constructor

		-- Define various types used by stylesheets
		PropertyTypes = PropertyTypesList
		Vec2 = Vec2Type
		Vec3 = Vec3Type
		Vec4 = Vec4Type
		Plane = Vec4Type
		Rect = RectType
		PivotMode = PivotModeType
		RgbColor  = RgbColorType
		RgbColour = RgbColorType
		HexColor  = HexColorType
		HexColour = HexColorType
		GrayscaleColor = GrayscaleColorType
		GreyscaleColour = GrayscaleColorType

		input =
		{
			Constants = InputConstants
		}
	end

	-- Loads the effects functions used when adding effects to nodes (Alpha etc)
	local function _loadEffects()

		-- Load the effects into the global namespace
		EffectLibrary.loadEffects()

	end

	-- Loads the animation functions used when adding animations to nodes (InSine etc)
	local function _loadAnimations()

		-- Load the animation functions into the global namespace
		AnimationLibrary.loadAnimationFunctions()

	end

	local function _loadIterators()
		IteratorsLibrary.loadIteratingFunctions()
	end

	-- Go!
	_initEnvironment()
	_defineKeywords()
	_loadEffects()
	_loadAnimations()
	_loadIterators()
end

init()
