-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local Counters = plugins.style:require('profiling.Counters')
local Counter = plugins.style:lazyRequire('profiling.Counter')
local CounterCategory = plugins.style:lazyRequire('profiling.CounterCategory')

local StatsManager = class(function(self)
	self._enableTimelining = false
	self._enableCounters = false
	self._counterCategories = {}
end)

function StatsManager:enableTimelining()
	self._enableTimelining = true
end

function StatsManager:disableTimelining()
	self._enableTimelining = false
end

function StatsManager:startTimelineAction(name)
	if self._enableTimelining then
		return timeline.startAction(name)
	end
end

-- Note that the `id` argument is optional
function StatsManager:stopTimelineAction(name, id)
	if self._enableTimelining then
		timeline.stopAction(name, id)
	end
end

function StatsManager:defineCounterCategory(name)
	local category = CounterCategory.new(name)
	table.insert(self._counterCategories, category)
	
	return category
end

function StatsManager:defineCounter(category, description, suffix)
	local counter = Counter.new(description, suffix)
	category:addCounter(counter)
	
	return counter
end

function StatsManager:resetCounters()
	for i, category in ipairs(self._counterCategories) do
		category:reset()
	end
end

function StatsManager:enableCounters()
	self._enableCounters = true
end

function StatsManager:disableCounters()
	self._enableCounters = false
end

function StatsManager:countersAreEnabled()
	return self._enableCounters
end

function StatsManager:printCounters()	
	print()
	print('Style Statistics')
	print('================')

	for i, category in ipairs(self._counterCategories) do
		if category:anyCountersAreGreaterThanZero() then
			print()
			print('  ' .. category:getName())
			print('  ----')
		
			for j, counter in ipairs(category:getCounters()) do
				print(
					'    ' .. 
					counter:getDescription() .. ': ' .. 
					counter:getCount() .. 
					counter:getSuffix())
			end
		end
	end
	
	print()
end

return StatsManager
