-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local Easing = plugins.style:lazyRequire('expressions.definitions.animations.Easing')
local ReflectableUtils = plugins.style:lazyRequire('utils.ReflectableUtils')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local StyleValueFormatter = plugins.style:lazyRequire('utils.StyleValueFormatter')

--
-- Class returned by AnimationRuntimeExpression:evaluate() so that PropertyStyle
-- or EffectStyle can start the resulting animation when ready.
--
local AnimationTrigger = class(function(self, node, property, from, to, duration, delay, parameters, easing, signalQueue, onCompleteSignalName)
	self._node = node
	self._property = property
	self._from = from
	self._to = to
	self._duration = duration
	self._delay = delay
	self._parameters = parameters
	self._easing = easing
	self._signalQueue = signalQueue
	self._onCompleteSignalName = onCompleteSignalName

end)

AnimationTrigger.DEFAULT_NUM_REPETITIONS = 0

-- Starts the animation via the animation plugin using animation.queue()
function AnimationTrigger:trigger()

	local propertyName = self._property:getName()

	-- Cancel any animations already operating on this property.
	self:cancel(propertyName)

	local parameters = (self._parameters and self._parameters:getValues()) or {}

	-- The animation plugin doesn't provide a 'delay' argument, but to simulate
	-- a delay we can start a nil animation (just tweens to the current value
	-- of the property) which will have the effect of delaying the start of the
	-- actual animation.
	local delay = self:_getValue(self._delay)
	if delay > 0 then
		animation.queue(self._node, propertyName, animation.type.FIXED_DURATION,
		{
			from = nil,
			to = nil,
			duration = delay,
			easingParams = parameters,
			easing = Easing.LINEAR,
			repetitions = 0
		})
	end

	-- Enqueue the actual animation to the target property value.
	local animationId = animation.queue(
		self._node, propertyName, animation.type.FIXED_DURATION,
		{
			from = self:_getValue(self._from),
			to = self:_getValue(self._to),
			duration = self:_getValue(self._duration),
			easingParams = parameters,
			easing = self._easing,
			repetitions = AnimationTrigger.DEFAULT_NUM_REPETITIONS
		})

	-- Subscribe to animation complete signal if a signal name has been set
	if (self._onCompleteSignalName ~= nil and animationId ~= nil) then
		self._signalQueue:queueSignal(self._node, self._onCompleteSignalName)

		animation.complete:add(function(uuid)
			if uuid == animationId then
				self._signalQueue:completeSignal(self._node, self._onCompleteSignalName)
			end
		end)

	end

end

-- Clears the animation via animation.clear()
function AnimationTrigger:cancel()
	self._signalQueue:dequeueSignal(self._node, self._onCompleteSignalName)
	animation.clear(self._node, self._property:getName())
end

function AnimationTrigger:getNode()
	return self._node
end

function AnimationTrigger:getProperty()
	return self._property
end

function AnimationTrigger:getFrom()
	return self._from
end

function AnimationTrigger:getTo()
	return self._to
end

function AnimationTrigger:getDuration()
	return self._duration
end

function AnimationTrigger:getDelay()
	return self._delay
end

function AnimationTrigger:getParameters()
	return self._parameters
end

function AnimationTrigger:getEasing()
	return self._easing
end

function AnimationTrigger:toString()
	return self._easing .. '(' ..
			StyleValueFormatter.formatValue(self:_getValue(self._to):get()) .. ', ' ..
			StyleValueFormatter.formatValue(self:_getValue(self._duration)) .. ')'
end

function AnimationTrigger:cloneWithNewToValue(to)
	return AnimationTrigger.new(
		self._node,
		self._property,
		self._from,
		ReflectableUtils.convertToReflectable(to),
		self._duration,
		self._delay,
		self._parameters,
		self._easing,
		self._signalQueue,
		self._onCompleteSignalName)
end

function AnimationTrigger.isAnimationTrigger(value)
	return getmetatable(value) == AnimationTrigger
end

function AnimationTrigger:_getValue(value)
	if AbstractRuntimeExpression.isExpression(value) then
		return value:evaluate(self._property, self._node)
	else
		return value
	end
end

return AnimationTrigger
