-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local AnimationRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AnimationRuntimeExpression')
local AnimationRuntimeExpressionBuilder = plugins.style:lazyRequire('expressions.definitions.builders.AnimationRuntimeExpressionBuilder')
local Easing = plugins.style:lazyRequire('expressions.definitions.animations.Easing')

local _getNamesToCapitalizedNames
local _getAnimationFunction

-- Creates an animation function for each Easing type listed in the easing
-- functions list. The naming of each generated function is the same as the
-- animation name but with the first letter capitalized, i.e. 'InSine'.
local function loadAnimationFunctions(into)

	local namespace = into or _G

	for easingName, capitalizedName in pairs(_getNamesToCapitalizedNames()) do

		-- Create the animation functions used during rule definition, e.g. InSine
		namespace[capitalizedName] = function(...)
			local args = {...}
			if args and #args > 0 then
				-- eg. OutQuad(1, 0.5, 0) -- for backwards compatibility
				return AnimationRuntimeExpression.new(
					nil,	 -- value to animate from. Nil means the `current` value
					args[1], -- value to animate to
					args[2], -- animation duration
					args[3], -- delay before beginning animation
					args[4], -- easing function parameters
					easingName)
			else
				-- eg. OutQuad():to(1.0):duration(0.5):delay(0) -- [RECOMMENDED]
				return AnimationRuntimeExpressionBuilder.new(easingName)
			end
		end

		-- Load the corresponding interpolation function, e.g. functions/inSine.lua
		if animation ~= nil then
			animation.loadFunction(easingName)
		end

	end

	-- Add the special 'NoTween' animation function, which is used for disabling
	-- the default behaviour of animations animating out when they are removed
	-- from a node
	namespace['NoTween'] = function(to)
		return AnimationRuntimeExpression.new(nil, to, 0, 0, nil, Easing.LINEAR)
	end

end

-- Reverses the action of loadAnimationFunctions(), either on the supplied
-- table or on the global namespace.
local function unloadAnimationFunctions(from)

	local namespace = from or _G

	for easingName, capitalizedName in pairs(_getNamesToCapitalizedNames()) do
		namespace[capitalizedName] = nil
	end

end

-- Returns a map containing all of the easing function names in the 'Easing' list,
-- mapped to a version of their name which has the first letter capitalized.
function _getNamesToCapitalizedNames()
	local namesToCapitalizedNames = {}

	for constantName, easingName in pairs(Easing) do
		local capitalized = easingName:sub(1, 1):upper() .. easingName:sub(2)
		namesToCapitalizedNames[easingName] = capitalized
	end

	return namesToCapitalizedNames
end

-- Public exports.
return
{
	loadAnimationFunctions = loadAnimationFunctions,
	unloadAnimationFunctions = unloadAnimationFunctions
}
