-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local RuntimeExpressionTypes = plugins.style:lazyRequire('expressions.RuntimeExpressionTypes')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local AnimationTrigger = plugins.style:lazyRequire('expressions.definitions.animations.AnimationTrigger')
local PropertyGetter = plugins.style:lazyRequire('rules.styles.definitions.PropertyGetter')

local Operators
local _operatorsToEvaluationFunctions
local _addOperatorOverloads

local EXPRESSION_TYPE = RuntimeExpressionTypes.MATHEMATICAL

--
-- RuntimeExpression created when mathematical operators are applied to the
-- results of other RuntimeExpressions. For example:
--
--    to "ImageNode.square" apply
--    {
--        x = previous.x + 100   <<< x must be calculated by reading the value
--                                   of previous.x at runtime, and then adding
--                                   100 to it.
--    }
--
local MathematicalRuntimeExpression = class(

	-- Parent class
	AbstractRuntimeExpression,

	-- Constructor
	function(self, lhs, rhs, operator)

		self:_init(EXPRESSION_TYPE)
		_addOperatorOverloads(self)

		if _operatorsToEvaluationFunctions[operator] == nil then
			error('Unrecognized operator: ' .. operator)

		elseif lhs == nil then
			error('LHS operand supplied to MathematicalRuntimeExpression was nil')

		elseif rhs == nil and operator ~= Operators.UNARY_MINUS then
			error('RHS operand supplied to MathematicalRuntimeExpression was nil')

		elseif operator == Operators.DIVIDE and AbstractRuntimeExpression.isExpression(rhs) then

			if rhs:getExpressionType() ~= RuntimeExpressionTypes.ZERO_GUARD then
				log.warn(
					'Attempting to divide value by a runtime expression, ' ..
					'please protect this call by wrapping it in a ZeroGuard')
			end
			
		end

		self._lhs = lhs
		self._rhs = rhs
		self._operator = operator
		self._evaluationFn = _operatorsToEvaluationFunctions[operator]

	end
)


-- Type constant identifying MathematicalRuntimeExpressions.
MathematicalRuntimeExpression.EXPRESSION_TYPE = EXPRESSION_TYPE

-- List of operators.
MathematicalRuntimeExpression.Operators =
{
	ADD = 'add',
	SUBTRACT = 'subtract',
	MULTIPLY = 'multiply',
	DIVIDE = 'divide',
	MODULO = 'modulo',
	POWER = 'power',
	CONCAT = 'concat',
	UNARY_MINUS = 'unary minus'
}

-- Local reference to the Operators map, to reduce unnecessary lookups.
Operators = MathematicalRuntimeExpression.Operators

-- Table mapping operator names to evaluation functions.
_operatorsToEvaluationFunctions =
{
	[Operators.ADD]         = function(a, b) return a + b  end,
	[Operators.SUBTRACT]    = function(a, b) return a - b  end,
	[Operators.MULTIPLY]    = function(a, b) return a * b  end,
	[Operators.DIVIDE]      = function(a, b) return a / b  end,
	[Operators.MODULO]      = function(a, b) return a % b  end,
	[Operators.POWER]       = function(a, b) return a ^ b  end,
	[Operators.CONCAT]      = function(a, b) return a .. b end,
	[Operators.UNARY_MINUS] = function(a)    return -a     end
}

-- Resolves the value of the runtime expression by applying the operator to
-- the operands and returning the result.
function MathematicalRuntimeExpression:evaluate(currentProperty, currentNode, phase)

	local lhs = self:_evaluateOperand(self._lhs, currentProperty, currentNode, phase)
	local rhs = self:_evaluateOperand(self._rhs, currentProperty, currentNode, phase)

	if self._operator == Operators.DIVIDE and rhs == 0 then
		log.error("Attempting to divide by zero, please use a ZeroGuard to avoid.")
		return 0
	end

	return self._evaluationFn(lhs, rhs)

end

-- PropertyGetter instance used by the _evaluateOperand() method below.
local _propertyGetter = PropertyGetter.new()

-- Resolves the value of the runtime expression by applying the operator to
-- the operands and returning the result.
function MathematicalRuntimeExpression:_evaluateOperand(
		operand,
		currentProperty,
		currentNode,
		phase)

	if (AbstractRuntimeExpression.isExpression(operand)) then
		local evaluator = self._evaluator
		local value = evaluator:evaluate(operand, currentProperty, currentNode, phase)

		-- If the value has returned an animation, we can't use it directly as
		-- the current value of the property will be dependent on whatever the
		-- animation plugin has set as the current value. In this case we have
		-- to read from the node directly.
		if AnimationTrigger.isAnimationTrigger(value) then
			local animatedNode = value:getNode()
			local animatedProperty = value:getProperty():getName()

			return _propertyGetter:getPropertyOfNode(animatedNode, animatedProperty)
		else
			return value
		end
	else
		return operand
	end

end

-- Adds overloads for each of Lua's mathematical operator metamethods which will
-- create a corresponding MathematicalRuntimeExpression when called.
function MathematicalRuntimeExpression.addOperatorOverloads(expression)

	local metatable = getmetatable(expression)

	if metatable == nil then
		metatable = {}
		setmetatable(expression, metatable)
	end

	metatable.__add = function(lhs, rhs)
		return MathematicalRuntimeExpression.new(lhs, rhs, Operators.ADD)
	end

	metatable.__sub = function(lhs, rhs)
		return MathematicalRuntimeExpression.new(lhs, rhs, Operators.SUBTRACT)
	end

	metatable.__mul = function(lhs, rhs)
		return MathematicalRuntimeExpression.new(lhs, rhs, Operators.MULTIPLY)
	end

	metatable.__div = function(lhs, rhs)
		return MathematicalRuntimeExpression.new(lhs, rhs, Operators.DIVIDE)
	end

	metatable.__mod = function(lhs, rhs)
		return MathematicalRuntimeExpression.new(lhs, rhs, Operators.MODULO)
	end

	metatable.__pow = function(lhs, rhs)
		return MathematicalRuntimeExpression.new(lhs, rhs, Operators.POWER)
	end

	metatable.__concat = function(lhs, rhs)
		return MathematicalRuntimeExpression.new(lhs, rhs, Operators.CONCAT)
	end

	metatable.__unm = function(lhs)
		return MathematicalRuntimeExpression.new(lhs, nil, Operators.UNARY_MINUS)
	end

end

-- Local reference to addOperatorOverloads so that we can refer to it from
-- the constructor.
_addOperatorOverloads = MathematicalRuntimeExpression.addOperatorOverloads

-- toString method used when printing errors related to this expression type.
-- Note that we can't use a __tostring metamethod as it won't survive the transfer
-- from the definition state to the application state.
function MathematicalRuntimeExpression:toString()

	local fullName = MathematicalRuntimeExpression.EXPRESSION_TYPE .. 'RuntimeExpression'
	local operatorDetails = 'for operator \'' .. self._operator .. '\''

	return fullName .. ' ' .. operatorDetails

end

function MathematicalRuntimeExpression:getLhs()
	return self._lhs
end

function MathematicalRuntimeExpression:getRhs()
	return self._rhs
end

function MathematicalRuntimeExpression:getOperator()
	return self._operator
end

-- Creates a table-based representation of the runtime expression, for use in
-- transferring the expression from the definition state to the application state.
function MathematicalRuntimeExpression:toTable()

	return
	{
		isExpressionTable = true,
		expressionType = MathematicalRuntimeExpression.EXPRESSION_TYPE,
		lhs = self._lhs,
		rhs = self._rhs,
		operator = self._operator,
	}

end

-- Performs the reverse of toTable(), for use in creating MathematicalRuntimeExpression
-- instances in the application state from data transferred from the definition state.
function MathematicalRuntimeExpression.fromTable(table)

	return MathematicalRuntimeExpression.new(
		table.lhs,
		table.rhs,
		table.operator)

end

-- Overrides setContext() to recursively set the context of sub-expressions.
function MathematicalRuntimeExpression:setContext(context)
	self._context = context
	self._evaluator = self._context:getExpressionEvaluator()

	if AbstractRuntimeExpression.isExpression(self._lhs) then
		self._lhs:setContext(context)
	end

	if AbstractRuntimeExpression.isExpression(self._rhs) then
		self._rhs:setContext(context)
	end
end

return MathematicalRuntimeExpression



