-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local StyleApplicationPhases = plugins.style:lazyRequire('rules.styles.StyleApplicationPhases')
local RuntimeExpressionTypes = plugins.style:lazyRequire('expressions.RuntimeExpressionTypes')
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')
local MathematicalRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.MathematicalRuntimeExpression')
local AnimationTrigger = plugins.style:lazyRequire('expressions.definitions.animations.AnimationTrigger')
local Easing = plugins.style:lazyRequire('expressions.definitions.animations.Easing')
local ReflectableUtils = plugins.style:lazyRequire('utils.ReflectableUtils')

local _assertNotNil
local _assertTypeOrNil
local _assertRecognizedEasingFunction
local _easingFunctionNames

local EXPRESSION_TYPE = RuntimeExpressionTypes.ANIMATION

--
-- RuntimeExpression created when animation functions are used as the value of
-- style properties. For example:
--
--    to "ImageNode.square" apply
--    {
--        x = InOutSine(100)   <<< x must be animated to 100 using an in-out
--                                 sine curve as the easing function.
--    }
--
local AnimationRuntimeExpression = class(

	-- Parent class
	AbstractRuntimeExpression,

	-- Constructor
	function(self, from, to, duration, delay, parameters, easing, onCompleteSignalName)

		self:_init(EXPRESSION_TYPE)
		MathematicalRuntimeExpression.addOperatorOverloads(self)

		_assertNotNil('to', to)
		_assertNotNil('duration', duration)
		_assertNotNil('easing', easing)
		_assertTypeOrNil('onCompleteSignalName', 'string', onCompleteSignalName)
		_assertTypeOrNil('parameters', 'table', parameters)
		_assertRecognizedEasingFunction(easing)

		self._from = from
		self._to = to
		self._duration = duration
		self._delay = delay or 0
		self._parameters = parameters
		self._easing = easing
		self._onCompleteSignalName = onCompleteSignalName

	end
)

-- Type constant identifying AnimationRuntimeExpressions.
AnimationRuntimeExpression.EXPRESSION_TYPE = EXPRESSION_TYPE

function AnimationRuntimeExpression:dispatchOnComplete(signalName)

	if type(signalName) == 'string' then		
		self._onCompleteSignalName = signalName
	else
		error(
			'The argument supplied to ' ..
			'AnimationRuntimeExpression:dispatchOnComplete() ' ..
			'must be a string'
		)
	end
	
	return self

end

-- Creates an AnimationTrigger capable of starting the animation when required.
function AnimationRuntimeExpression:evaluate(currentProperty, currentNode, phase)

	local to = self:_getReflectableFromExpressionOrValue(
		self._to, currentProperty, currentNode, phase)
	
	local from = self:_getReflectableFromExpressionOrValue(
		self._from, currentProperty, currentNode, phase)
	
	local property = self._associatedProperty

	return AnimationTrigger.new(
		currentNode,
		property,
		from,
		to,
		self._duration,
		self._delay,
		self._parameters,
		self._easing,
		self._context:getSignalQueue(),
		self._onCompleteSignalName)

end

-- Works out whether the value is an expression, and if so evaluates it.
-- Also converts it to a reflectable, which is what animation.queue() requires.
function AnimationRuntimeExpression:_getReflectableFromExpressionOrValue(
	value, currentProperty, currentNode, phase)

	local evaluator = self._context:getExpressionEvaluator()

	if AbstractRuntimeExpression.isExpression(value) then
		value = evaluator:evaluate(value, currentProperty, currentNode, phase)
	end

	return ReflectableUtils.convertToReflectable(value)

end

-- toString method used when printing errors related to this expression type.
-- Note that we can't use a __tostring metamethod as it won't survive the transfer
-- from the definition state to the application state.
function AnimationRuntimeExpression:toString()

	local fullName = AnimationRuntimeExpression.EXPRESSION_TYPE .. 'RuntimeExpression'
	local easingDetails = 'for easing function \'' .. self._easing .. '\','
	local durationDetails = 'duration ' .. self._duration .. ' and delay ' .. self._delay

	return fullName .. ' ' .. easingDetails .. ' ' .. durationDetails

end

function AnimationRuntimeExpression:getFrom()
	return self._from
end

function AnimationRuntimeExpression:getTo()
	return self._to
end

function AnimationRuntimeExpression:getDuration()
	return self._duration
end

function AnimationRuntimeExpression:getDelay()
	return self._delay
end

function AnimationRuntimeExpression:getParameters()
	return self._parameters
end

function AnimationRuntimeExpression:getEasing()
	return self._easing
end

-- Creates a table-based representation of the runtime expression, for use in
-- transferring the expression from the definition state to the application state.
function AnimationRuntimeExpression:toTable()

	return
	{
		isExpressionTable = true,
		expressionType = AnimationRuntimeExpression.EXPRESSION_TYPE,
		from = self._from,
		to = self._to,
		duration = self._duration,
		delay = self._delay,
		parameters = self._parameters,
		easing = self._easing,
		onCompleteSignalName = self._onCompleteSignalName
	}

end

-- Performs the reverse of toTable(), for use in creating AnimationRuntimeExpression
-- instances in the application state from data transferred from the definition state.
function AnimationRuntimeExpression.fromTable(table)

	return AnimationRuntimeExpression.new(
		table.from,
		table.to,
		table.duration,
		table.delay,
		table.parameters,
		table.easing,
		table.onCompleteSignalName)

end


-- Utility functions

function _assertNotNil(argumentName, argumentValue)
	if argumentValue == nil then
		error(
			'The \'' .. argumentName .. '\' argument supplied to an ' ..
			'AnimationRuntimeExpression must not be nil'
		)
	end
end

function _assertRecognizedEasingFunction(easingFunctionName)
	if not _easingFunctionNames[easingFunctionName] then
		error('Easing function \'' .. easingFunctionName .. '\' was not recognized')
	end
end

function _assertTypeOrNil(argumentName, argumentType, argumentValue)
	if type(argumentValue) ~= argumentType then
		if argumentValue ~= nil then
			error(
				'The \'' .. argumentName .. '\' argument supplied to an ' ..
				'AnimationRuntimeExpression must be a ' .. argumentType
			)
		end
	end
end

-- Builds the map used in the above assertion function
(function()
	_easingFunctionNames = {}

	for constantName, functionName in pairs(Easing) do
		_easingFunctionNames[functionName] = true
	end
end)()


return AnimationRuntimeExpression



