-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local ffi = lazyRequire('ffi')

local PropertyTypes = plugins.style:lazyRequire('rules.styles.definitions.PropertyTypes')
local PropertyTypeChecker = plugins.style:lazyRequire('rules.styles.definitions.PropertyTypeChecker')

local VALID_TYPES = 
{
	PropertyTypes.INT,
	PropertyTypes.UINT,
	PropertyTypes.FLOAT,
	PropertyTypes.STRING,
	PropertyTypes.BOOLEAN
	-- TODO: Add support for Vec2/3/4?
}

local Bridge = class(function(self)
    self:clear()
end)

function Bridge:clear()
    self._variables = {}
    self._variablesType = {}
    self._variableSize = 0
end

function Bridge:getVariables()
    return self._variables
end

function Bridge:getVariable(variableName)
    return self._variables[variableName]
end

function Bridge:getVariableType(variableName)
    return self._variablesType[variableName]
end

function Bridge:getNumVariables()
    return self._variableSize
end

function Bridge:exportVariables()
    local bridgeObjectContainer =
    {
        variables = self._variables,
        types = self._variablesType,
        numVariables = self._variableSize,
    }

    return bridgeObjectContainer
end

function Bridge:importVariables(bridgeObjectContainer)
    for k,v in pairs(bridgeObjectContainer.variables) do
        self._variables[k] = v
    end
    for k,v in pairs(bridgeObjectContainer.types) do
        self._variablesType[k] = v
    end

    local numVars = 0
    for k,v in pairs(self._variables) do
        numVars = numVars + 1
    end

    self._variableSize = numVars
end

function Bridge:isAVariableType(variableType)
    local errorMsg

    if type(variableType) == 'string' then
        for k, varType in pairs(VALID_TYPES) do
            if variableType == varType then
                return true
            end
        end
        errorMsg = "Variable type is not recognised : '"..variableType.."'"
    else
        errorMsg = "Variable type is not recognised." 
    end

    errorMsg = errorMsg.."\nPlease use one type in the following list :" 
    for k, varType in pairs(VALID_TYPES) do
        errorMsg = errorMsg.."\n - '"..varType.."'"
    end

    error(errorMsg)
end

function Bridge:declareVariable(variableName, variableType, initialValue)

    if self:isAVariableType(variableType) then
        self._variablesType[variableName] = variableType
        self:setVariable(variableName, initialValue)
    end
end

local CONST_CHAR_CTYPE = ffi.typeof('char *')

function Bridge:setVariable(variableName, variableValue)

    if self._variables[variableName] == nil then
        self._variableSize = self._variableSize + 1
    end

    if variableValue == nil then
       self._variableSize = self._variableSize - 1 
    end

    -- Check if the variable name has been declared 
    if self._variablesType[variableName] == nil then
        error("Trying to set variable '"..variableName.."' that has not been declared")
	end

	-- Strings that come from native will come in as userdata - we need to
	-- convert it to an actual Lua string via ffi.string()
	if type(variableValue) == 'userdata' then
		variableValue = ffi.string(variableValue)
	end

    -- Perform a type checking
	local expectedType = self._variablesType[variableName]
    local mismatchedType = PropertyTypeChecker.checkForTypeMismatch(
		expectedType,
		variableValue)

    if mismatchedType ~= nil then
        error("Trying to set variable '"..variableName
            .."' of type '"..expectedType
            .."' with type '"..mismatchedType.."'")
    end

    self._variables[variableName] = variableValue
end

function Bridge:printTable()
    print("Bridge._variable {")
    for k, v in pairs(self._variables) do
        print("\t"..k.." : "..v)
    end
    print("}")
end

return Bridge
