-- Copyright 2014 Amazon.com, Inc. or its affiliates. All Rights Reserved.
local AbstractRuntimeExpression = plugins.style:lazyRequire('expressions.definitions.AbstractRuntimeExpression')

--
-- Class used by RuleApplier when attempting to evaluate the result of a runtime
-- expression used in a a rule. Basically provides a map of expression types to
-- the expression class associated with it, so that RuleApplier can just call
-- ExpressionEvaluator.evaluate(expression, node) without caring exactly what
-- type of expression has been supplied.
--
local ExpressionEvaluator = class(function(self, context)

	self._context = context
	self._activeStyleRegistry = context:getStyleRegistry()
	self._bindingManager = context:getBindingManager()
	self._bridge = context:getBridge()
	self._expressionClassesByType = {}

end)

local _evaluate
local _throwCircularExpressionError

function ExpressionEvaluator:loadExpressionType(expressionClass)
	self._expressionClassesByType[expressionClass.EXPRESSION_TYPE] = expressionClass
end

-- Evaluates the supplied expression, returning the value. If the expression
-- returns another expression, this function will continue to recurse until a
-- non-expression value is returned.
--
-- Some checking is provided to ensure that loops are not encountered in the
-- expressions tree - if two expressions reference each other (even if it's
-- via several layers of indirection), an error will be thrown in order to
-- prevent infinite recursion.
function ExpressionEvaluator:evaluate(expression, currentProperty, currentNode, phase)

	return _evaluate(expression, currentProperty, currentNode, phase, {})

end

function _evaluate(expression, currentProperty, currentNode, phase, visited)

	local result = expression:evaluate(currentProperty, currentNode, phase)

	if AbstractRuntimeExpression.isExpression(result) then
		-- Check that we haven't encountered a loop (see the marking line below)
		if visited[expression] then
			_throwCircularExpressionError(expression)
		end

		-- Mark that we've visited this expression already, so that we can detect
		-- circular expression trees
		visited[expression] = true

		return _evaluate(result, currentProperty, currentNode, phase, visited)
	else
		return result
	end

end

function _throwCircularExpressionError(expression)

	local errorMessage = 'Circular expression detected whilst evaluating '
	local expressionInfo = expression:toString(expression)

	error(errorMessage .. expressionInfo)

end

-- Creates a new RuntimeExpression instance based on the given expressionTable,
-- which will have been supplied from the rule definition state.
function ExpressionEvaluator:createExpressionFromTable(table)

	local expressionClass = self._expressionClassesByType[table.expressionType]
	local expressionInstance = expressionClass.fromTable(table)

	expressionInstance:setContext(self._context)

	return expressionInstance

end

return ExpressionEvaluator