-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local MutationTrackingData = class(function(self, sceneNodes, mutations)
	self._startTime = -1
	self._endTime = -1
	self._sceneNodes = sceneNodes
	self._mutations = mutations
	self._sceneNodesToProgressFlags = {}
	self._sceneNodesToBindingUpdates = {}
	self._sceneNodesToStyleApplications = {}
	self._nodeMutationInfos = {}
end)

function MutationTrackingData:mutationsTookPlace()	
	return self:getNumNodes() > 0 and self:getNumMutations() > 0
end

function MutationTrackingData:bindingUpdatesTookPlace()	
	local updatesTookPlace = false
	for i, v in pairs(self._sceneNodesToBindingUpdates) do
		updatesTookPlace = true
		break
	end
	return updatesTookPlace
end

function MutationTrackingData:getNumNodes()	
	return #self._sceneNodes
end

function MutationTrackingData:getNumMutations()	
	return #self._mutations
end

function MutationTrackingData:getSceneNodes()	
	return self._sceneNodes
end

function MutationTrackingData:getMutations()	
	return self._mutations
end

function MutationTrackingData:recordStartTime()
	self._startTime = timer.getTimeInMilliseconds()
end

function MutationTrackingData:recordEndTime()
	self._endTime = timer.getTimeInMilliseconds()
end

function MutationTrackingData:getStartTime()
	return self._startTime
end

function MutationTrackingData:getEndTime()
	return self._endTime
end

function MutationTrackingData:getDuration()	
	if self._startTime ~= -1 and self._endTime ~= -1 then
		return self._endTime - self._startTime
	else
		error('Cannot calculate duration unless startTime and endTime are both known')
	end
end

function MutationTrackingData:getNodeMutationInfos()	
	return self._nodeMutationInfos
end

function MutationTrackingData:addNodeMutationInfo(nodeMutationInfo)	
	table.insert(self._nodeMutationInfos, nodeMutationInfo)
end

function MutationTrackingData:getProgressFlagsForNode(node)
	return self:_getOrCreateBucketForNode(self._sceneNodesToProgressFlags, node)
end

function MutationTrackingData:getStyleApplicationsForNode(node)
	return self:_getOrCreateBucketForNode(self._sceneNodesToStyleApplications, node)
end

function MutationTrackingData:_getOrCreateBucketForNode(container, node)
	local nodeId = node:getId()
	if container[nodeId] == nil then
		container[nodeId] = {}
	end
	return container[nodeId]
end

function MutationTrackingData:flagProgressForNode(node, stage)
	assert(node ~= nil,  "The node supplied to flagProgressForNode() cannot be nil")
	assert(stage ~= nil, "The stage supplied to flagProgressForNode() cannot be nil")

	table.insert(self:getProgressFlagsForNode(node), stage)
end

function MutationTrackingData:flagStyleApplicationForNode(node, style, value)
	assert(node ~= nil,  "The node supplied to flagStyleApplicationForNode() cannot be nil")
	assert(style ~= nil, "The style supplied to flagStyleApplicationForNode() cannot be nil")

	table.insert(self:getStyleApplicationsForNode(node), 
	{
		style = style,
		value = value
	})
end

function MutationTrackingData:getBindingUpdatesForNode(node)
	return self:_getOrCreateBucketForNode(self._sceneNodesToBindingUpdates, node)
end

function MutationTrackingData:recordBindingUpdateForNode(node, updateInfo)
	assert(node ~= nil,  "The node supplied to recordBindingUpdateForNode() cannot be nil")
	assert(updateInfo ~= nil, "The updateInfo supplied to recordBindingUpdateForNode() cannot be nil")

	table.insert(self:getBindingUpdatesForNode(node), updateInfo)
end

function MutationTrackingData:getProgressFlagCounts()
	local counts = {}

	for nodeId, progressFlags in pairs(self._sceneNodesToProgressFlags) do
		for i, flag in ipairs(progressFlags) do
			if counts[flag] == nil then
				counts[flag] = 1
			else
				counts[flag] = counts[flag] + 1
			end
		end
	end
	
	return counts
end

return MutationTrackingData