-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local MutationTrackingData = plugins.style:lazyRequire('utils.mutationGraph.MutationTrackingData')
local PrintableNodeMutationInfo = plugins.style:lazyRequire('utils.mutationGraph.PrintableNodeMutationInfo')
local PrintableBindingUpdateInfo = plugins.style:lazyRequire('utils.mutationGraph.PrintableBindingUpdateInfo')

local MutationTracker = class(function(self, matchRegistry, styleRegistry)
	self._matchRegistry = matchRegistry
	self._styleRegistry = styleRegistry
	self._isEnabled = false
	self._trackingIsInProgress = false
	self._data = nil
end)

function MutationTracker:enable()
	self._isEnabled = true
end

function MutationTracker:disable()
	self._isEnabled = false
end

function MutationTracker:beginTracking(ruleSet, sceneNodes, mutations)
	if not self._isEnabled then
		return
	end

	self._trackingIsInProgress = true
	self._ruleSet = ruleSet
	self._data = MutationTrackingData.new(sceneNodes, mutations)
	self._data:recordStartTime()
end

function MutationTracker:endTracking()
	if not self._isEnabled then
		return
	end
	
	if not self._trackingIsInProgress then
		error('Cannot end tracking when tracking is not already in progress.')
	end
	
	self._trackingIsInProgress = false
	self._data:recordEndTime()
	self:_generateNodeMutationInfos()
	
	return self._data
end

function MutationTracker:trackingIsInProgress()
	return self._trackingIsInProgress
end

function MutationTracker:flagProgressForNode(node, stage)
	if not self._trackingIsInProgress then
		return
	end
	
	self._data:flagProgressForNode(node, stage)
end

function MutationTracker:flagStyleApplicationForNode(node, style)
	if not self._trackingIsInProgress then
		return
	end
	
	local value = self._styleRegistry:getLastAppliedValue(node, style)
	
	self._data:flagStyleApplicationForNode(node, style, value)
end

function MutationTracker:recordBindingUpdate(
		sourceNode, 
		sourceProperty,
		targetNode, 
		targetProperty,
		value)
	if not self._trackingIsInProgress then
		return
	end
	
	self._data:recordBindingUpdateForNode(sourceNode, PrintableBindingUpdateInfo.new(
			sourceNode, 
			sourceProperty,
			targetNode, 
			targetProperty,
			value))
end

function MutationTracker:_generateNodeMutationInfos()	
	if self._data:mutationsTookPlace() then
		self:_retrieveRootSceneNodeReference()
		self:_createSceneNodesToMutationsMap()
		self:_recursivelyGenerateNodeMutationInfos(self._rootSceneNode)
	end
end

-- TODO: This is a pretty hacky way to get a reference to the root node of the
-- scene graph. It'd be better to add an FFI binding for SceneGraph:getRootNode().
--
-- As this is just a debug tool though, and this method only gets called once, 
-- it's not really a big deal.
function MutationTracker:_retrieveRootSceneNodeReference()
	if self._rootSceneNode == nil then
		local node = self._data:getSceneNodes()[1]
		local parent = node:getParent()
		while parent ~= nil do
			node = parent
			parent = node:getParent()
		end
		
		self._rootSceneNode = node
	end
end

function MutationTracker:_createSceneNodesToMutationsMap()
	local sceneNodes = self._data:getSceneNodes()
	local mutations = self._data:getMutations()
	
	self._sceneNodesToMutations = {}

	for i, sceneNode in ipairs(sceneNodes) do
		self._sceneNodesToMutations[sceneNode:getId()] = mutations[i]
	end
end

function MutationTracker:_recursivelyGenerateNodeMutationInfos(sceneNode)
	local mutations = self._sceneNodesToMutations[sceneNode:getId()] 
	local progressFlags = self._data:getProgressFlagsForNode(sceneNode)
	local bindingUpdates = self._data:getBindingUpdatesForNode(sceneNode)
	local styleApplications = self._data:getStyleApplicationsForNode(sceneNode)
	local matches = self._matchRegistry:getMatchesForNode(sceneNode)

	self._data:addNodeMutationInfo(PrintableNodeMutationInfo.new(
			sceneNode, 
			mutations,
			progressFlags,
			bindingUpdates,
			styleApplications,
			matches,
			self._ruleSet))
	
	for i = 0, sceneNode:getNumberOfChildren() - 1 do
		self:_recursivelyGenerateNodeMutationInfos(sceneNode:getChildAtIndex(i))
	end
end

return MutationTracker

