-- Copyright 2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.

local DirectoryUtils = plugins.style:lazyRequire('utils.DirectoryUtils')
local FileWriter = plugins.style:lazyRequire('utils.FileWriter')

local MutationGraphPrinter = class(function(self, directoryUtils, fileWriter)
	self._directoryUtils = directoryUtils or DirectoryUtils.new()
	self._fileWriter = fileWriter or FileWriter.new()
end)

function MutationGraphPrinter:printGraph(mutationTrackingData, tickIndex)
	if mutationTrackingData and mutationTrackingData:mutationsTookPlace() then
		self:_writeFile(self:_createGraphvizString(mutationTrackingData), tickIndex)
	end
end

function MutationGraphPrinter:_createGraphvizString(mutationTrackingData)
	local nodeMutationInfos = mutationTrackingData:getNodeMutationInfos()
	local graphvizStrings = {}
	
	table.insert(graphvizStrings, 'digraph {')
	table.insert(graphvizStrings, '\tgraph [ fontsize=14 fontname="Verdana" compound=true ];')
	table.insert(graphvizStrings, '')

	table.insert(graphvizStrings, self:_createComment('Scene graph nodes'))
	table.insert(graphvizStrings, self:_createNodes(nodeMutationInfos))
	table.insert(graphvizStrings, '')

	table.insert(graphvizStrings, self:_createComment('Scene graph edges'))
	table.insert(graphvizStrings, self:_createEdges(nodeMutationInfos))
	table.insert(graphvizStrings, '')
	
	if mutationTrackingData:bindingUpdatesTookPlace() then
		table.insert(graphvizStrings, self:_createComment('Binding update edges'))
		table.insert(graphvizStrings, self:_createBindingUpdatesLinks(nodeMutationInfos))
		table.insert(graphvizStrings, '')
	end
	
	table.insert(graphvizStrings, self:_createComment('Statistics panel'))
	table.insert(graphvizStrings, self:_createStatisticsPanel(mutationTrackingData))
	table.insert(graphvizStrings, '}')
	
	return table.concat(graphvizStrings, '\n')
end

function MutationGraphPrinter:_createComment(comment)
	return '\t// ' .. comment
end

function MutationGraphPrinter:_createNodes(nodeMutationInfos)
	local strings = {}
	
	for i, mutationInfo in ipairs(nodeMutationInfos) do
		table.insert(strings, '\t' .. mutationInfo:getGraphNodeString())
	end
	
	return table.concat(strings, '\n')
end

function MutationGraphPrinter:_createEdges(nodeMutationInfos)
	local strings = {}
	
	table.insert(strings, 
			'\tedge[ arrowhead=none ];')
	
	for i, mutationInfo in ipairs(nodeMutationInfos) do
		local parentage = mutationInfo:getParentageString()
		
		if parentage ~= nil then
			table.insert(strings, '\t' .. parentage)
		end
	end
	
	return table.concat(strings, '\n')
end

function MutationGraphPrinter:_createBindingUpdatesLinks(nodeMutationInfos)
	local strings = {}
	
	table.insert(strings, 
			'\tedge[ arrowhead=normal constraint=false color=gray48 fontcolor=gray48 style=dashed ];')

	for i, mutationInfo in ipairs(nodeMutationInfos) do
		for j, bindingInfo in ipairs(mutationInfo:getBindingUpdates()) do
			table.insert(strings, '\t' .. bindingInfo:getEdgeString())
		end
	end
	
	return table.concat(strings, '\n')
end

function MutationGraphPrinter:_createStatisticsPanel(mutationTrackingData)
	local progressFlagCounts = mutationTrackingData:getProgressFlagCounts()
	local nodeMutationInfos = mutationTrackingData:getNodeMutationInfos()
	local strings = {}
	
	table.insert(strings, '\t{ rank = min;')
	table.insert(strings, '\t\tLegend [shape=none, margin=0, label=<')
	table.insert(strings, '\t\t<table border="0" cellborder="1" cellspacing="0" cellpadding="4">')
	
	self:_pushStatisticTitle(strings, 'Statistics')
	self:_pushStatisticRow(strings, 'Duration', mutationTrackingData:getDuration() .. 'ms')
	self:_pushStatisticRow(strings, 'Total nodes', #nodeMutationInfos)
	self:_pushStatisticRow(strings, 'Mutated nodes', mutationTrackingData:getNumNodes())
	
	self:_pushStatisticTitle(strings, 'Progress counts')
	for progressFlagName, count in pairs(progressFlagCounts) do
		self:_pushStatisticRow(strings, ' ' .. progressFlagName, count)
	end
	
	table.insert(strings, '\t\t</table>>];')
	table.insert(strings, '\t}')

	return table.concat(strings, '\n')
end

function MutationGraphPrinter:_pushStatisticTitle(strings, title)
	table.insert(strings, '\t\t\t' .. 
		'<tr>' .. 
			'<td colspan="2"><b>' .. title .. '</b></td>' ..
		'</tr>')
end

function MutationGraphPrinter:_pushStatisticRow(strings, name, value)
	table.insert(strings, '\t\t\t' .. 
		'<tr>' .. 
			'<td align="left">' .. name .. '</td>' .. 
			'<td>' .. value .. '</td>' .. 
		'</tr>')
end

function MutationGraphPrinter:_writeFile(content, tickIndex)
	local varDirectoryPath = self:_getVarDirectoryPath()

	if self._directoryUtils:ensureDirectoryExists(varDirectoryPath) then
		local name = 'mutation_graph_' .. tickIndex .. '.dot'
		self._fileWriter:writeFile(varDirectoryPath .. name, content)
	end
end

local VAR_DIRECTORY = 'user://com.amazon.ignition.framework.style/'

function MutationGraphPrinter:_getVarDirectoryPath(filename)
	local fullPath = VAR_DIRECTORY .. (filename or '')
	return plugins.resolvePath(fullPath) or fullPath
end

return MutationGraphPrinter

